import { test, expect } from '@playwright/test';

// Configure tests to run sequentially to avoid session conflicts
test.describe.configure({ mode: 'serial' });

test.describe('Backend Tag Page Tests', () => {
  test.beforeEach(async ({ page, context }) => {
    try {
      // Clear cookies and storage to ensure fresh session
      await context.clearCookies();
      await context.clearPermissions();
      
      // Navigate to the admin tag page with increased timeout
      await page.goto('http://127.0.0.1:8000/admin/tag', { timeout: 60000 });
      
      // Wait for login form to be visible
      await page.waitForSelector('input[name="username"]', { timeout: 30000 });
      
      // Wait for page to load
      await page.waitForLoadState('networkidle', { timeout: 30000 });
      
      // Check if we need to login
      const loginHeading = page.getByRole('heading', { name: 'Login' });
      const isLoginPage = await loginHeading.isVisible({ timeout: 5000 }).catch(() => false);
      
      if (isLoginPage) {
        console.log('Login form detected, attempting login...');
        
        // Fill login form
        await page.getByRole('textbox', { name: 'Username' }).fill('admin');
        await page.getByRole('textbox', { name: 'Password' }).fill('admin');
        await page.getByRole('button', { name: 'Login' }).click();
        
        // Wait for response
        await page.waitForTimeout(2000);
        await page.waitForLoadState('networkidle', { timeout: 15000 });
        
        // Navigate to the tag page after login
        await page.goto('http://127.0.0.1:8000/admin/tag', { timeout: 30000 });
        await page.waitForLoadState('networkidle', { timeout: 15000 });
      }
    } catch (error) {
      console.log('Navigation error, retrying...', error.message);
      // Check if page is still valid before retrying
      if (!page.isClosed()) {
        try {
          await page.goto('http://127.0.0.1:8000/admin/tag', { timeout: 30000 });
          await page.waitForLoadState('networkidle', { timeout: 15000 });
        } catch (retryError) {
          console.log('Retry failed, continuing with test...', retryError.message);
        }
      }
    }
  });

  test('should access admin tag page successfully', async ({ page }) => {
    try {
      // Check if the page title contains expected text
      await expect(page).toHaveTitle(/Z\.com/);
      
      // Check if the URL is correct (allow for redirects)
      const currentUrl = page.url();
      expect(currentUrl).toMatch(/\/admin\/tag|redirect.*tag/);
      
      // Wait for page to fully load
      await page.waitForTimeout(2000);
        
      // Check basic page elements exist (without strict visibility requirements)
      const searchBox = page.locator('input[type="search"], input[placeholder*="Search"], [role="searchbox"]').first();
      const addButton = page.locator('a:has-text("Add"), button:has-text("Add"), [href*="add"]').first();
      const table = page.locator('table, [role="table"]').first();
      
      // Just check if elements exist, not if they're visible
      await expect(searchBox).toBeAttached({ timeout: 10000 });
      await expect(addButton).toBeAttached({ timeout: 10000 });
      await expect(table).toBeAttached({ timeout: 10000 });
      
    } catch (error) {
      console.log('Test error:', error.message);
      // Take screenshot for debugging
      await page.screenshot({ path: 'test-failure.png' });
      throw error;
    }
  });

  test('should test basic page functionality', async ({ page }) => {
    // Set longer timeout for this test
    test.setTimeout(60000);
    
    try {
      // Wait for page to load
      await page.waitForLoadState('networkidle', { timeout: 30000 });
      
      // Check if we need to login
      const loginHeading = page.getByRole('heading', { name: 'Login' });
      const isLoginPage = await loginHeading.isVisible({ timeout: 10000 });
      
      if (isLoginPage) {
        console.log('Login form detected, attempting login...');
        
        // Fill login form with error handling
        try {
          await page.getByRole('textbox', { name: 'Username' }).fill('admin');
          await page.getByRole('textbox', { name: 'Password' }).fill('admin');
          await page.getByRole('button', { name: 'Login' }).click();
          
          // Wait for redirect
          await page.waitForTimeout(8000);
        } catch (error) {
          console.log('Login form interaction error:', error.message);
        }
      }
      
      // Verify we're on the admin tag page
      await expect(page).toHaveURL(/.*admin.*tag.*/, { timeout: 15000 });
      await expect(page).toHaveTitle('Z.com');
      
      // Wait for page to fully load
      await page.waitForTimeout(5000);
      
      // Test search functionality if search box is visible
      const searchBox = page.locator('input[type="search"], input[placeholder*="Search"], [role="searchbox"]').first();
      if (await searchBox.isVisible({ timeout: 10000 })) {
        await searchBox.fill('test');
        await page.waitForTimeout(3000);
      }
      
      // Test if we can find any links on the page
      const links = page.locator('a');
      const linkCount = await links.count();
      expect(linkCount).toBeGreaterThan(0);
      
      // Test if we can find any buttons on the page
      const buttons = page.locator('button');
      const buttonCount = await buttons.count();
      expect(buttonCount).toBeGreaterThan(0);
      
    } catch (error) {
      console.log('Test error:', error.message);
      await page.screenshot({ path: 'test-basic-functionality-failure.png' });
      throw error;
    }
  });

  test('should test page navigation', async ({ page }) => {
    // Set longer timeout for this test
    test.setTimeout(60000);
    
    try {
      // Wait for page to load
      await page.waitForLoadState('networkidle', { timeout: 30000 });
      
      // Check if we need to login
      const loginHeading = page.getByRole('heading', { name: 'Login' });
      const isLoginPage = await loginHeading.isVisible({ timeout: 10000 });
      
      if (isLoginPage) {
        console.log('Login form detected, attempting login...');
        
        // Fill login form with error handling
        try {
          await page.getByRole('textbox', { name: 'Username' }).fill('admin');
          await page.getByRole('textbox', { name: 'Password' }).fill('admin');
          await page.getByRole('button', { name: 'Login' }).click();
          
          // Wait for redirect
          await page.waitForTimeout(8000);
        } catch (error) {
          console.log('Login form interaction error:', error.message);
        }
      }
      
      // Verify we're on the admin tag page
      await expect(page).toHaveURL(/.*admin.*tag.*/, { timeout: 15000 });
      await expect(page).toHaveTitle('Z.com');
      
      // Wait for page to fully load
      await page.waitForTimeout(5000);
      
      // Test if we can navigate to other admin pages
      const settingLink = page.locator('a[href*="admin/setting"]').first();
      if (await settingLink.isVisible({ timeout: 10000 })) {
        await settingLink.click();
        await expect(page).toHaveURL(/.*admin.*setting.*/, { timeout: 15000 });
        
        // Navigate back to tag page
        await page.goto('http://127.0.0.1:8000/admin/tag');
        await page.waitForLoadState('networkidle', { timeout: 20000 });
      }
      
    } catch (error) {
      console.log('Test error:', error.message);
      await page.screenshot({ path: 'test-navigation-failure.png' });
      throw error;
    }
  });
});
