import { test, expect } from '@playwright/test';

test.describe('PDPA Admin Tests', () => {
    const BASE_URL = process.env.BASE_URL || 'http://127.0.0.1:8000';

    async function login(page) {
        await page.goto(`${BASE_URL}/admin`);
        await page.getByRole('textbox', { name: 'Username' }).fill('admin');
        await page.getByRole('textbox', { name: 'Password' }).fill('admin');
        await page.getByRole('button', { name: 'Login' }).click();
    }

    async function DetailPolicy(page) {
        await page.getByRole('link', { name: ' Policy 󰅂' }).click();
        await page.getByRole('link', { name: 'detail' }).click();
        await page.waitForLoadState('networkidle');
    }

    test.beforeEach(async ({ page }) => {
        // Navigate to PDPA admin page
        await login(page);
    });

    test.describe('Name Field Tests', () => {
            test('กรอกชื่อ - Should allow entering name', async ({ page }) => {
                // Click add new PDPA button
                await DetailPolicy(page);

                // await expect(page.locator('[id="105"]')).toContainText('Information');
                // await page.locator('[id="105"]').getByRole('listitem').filter({ hasText: 'Information' }).click();
                // Fill in name field
                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('นโยบายความเป็นส่วนตัวทดสอบ');

                await page.getByRole('combobox', { name: 'English' }).click();
                await page.getByRole('option', { name: 'ภาษาไทย' }).locator('div').first().click();

                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('Thai Privacy Policy Test');

                // Verify name is entered
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
            });

            test('แก้ไขชื่อ - Should allow editing name', async ({ page }) => {
                // Click add new PDPA button
                await DetailPolicy(page);

                // Fill in name field
                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('นโยบายความเป็นส่วนตัวทดสอบ-อัปเดต');

                await page.getByRole('combobox', { name: 'English' }).click();
                await page.getByRole('option', { name: 'ภาษาไทย' }).locator('div').first().click();

                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('Thai Privacy Policy Test-update');

                // Verify name is entered
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
            });

            test('กรอกชื่อ และอักขระพิเศษ - Should allow special characters in name', async ({ page }) => {
                // Click add new PDPA button
                await DetailPolicy(page);

                // Fill in name field
                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('นโยบายความเป็นส่วนตัวทดสอบ-!@#$%^&*()_+-=[]{}|;:,.<>?');

                await page.getByRole('combobox', { name: 'English' }).click();
                await page.getByRole('option', { name: 'ภาษาไทย' }).locator('div').first().click();

                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('Thai Privacy Policy Test-!@#$%^&*()_+-=[]{}|;:,.<>?');

                // Verify name is entered
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
            });

            test('แก้ไขชื่อ และอักขระพิเศษ - Should allow editing name with special characters', async ({ page }) => {
                // Click add new PDPA button
                await DetailPolicy(page);

                // Fill in name field
                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('อัปเดตนโยบายความเป็นส่วนตัวทดสอบ-!@#$%^&*()_+-=[]{}|;:,.<>?');

                await page.getByRole('combobox', { name: 'English' }).click();
                await page.getByRole('option', { name: 'ภาษาไทย' }).locator('div').first().click();

                await expect(page.locator('#info')).toContainText('Name');
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('update Thai Privacy Policy Test-!@#$%^&*()_+-=[]{}|;:,.<>?');

                // Verify name is entered
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');

            });

            test('กรอกรายละเอียดเกิน 255 ตัว - Should handle long names', async ({ page }) => {
                  await DetailPolicy(page);

                  const longName = 'น'.repeat(300); // 300 characters
                  // Fill in name field
                  await expect(page.locator('#info')).toContainText('Name');
                  await page.getByRole('textbox', { name: 'Please specify Name' }).fill(longName);

                  await page.getByRole('combobox', { name: 'English' }).click();
                  await page.getByRole('option', { name: 'ภาษาไทย' }).locator('div').first().click();

                  await expect(page.locator('#info')).toContainText('Name');
                  await page.getByRole('textbox', { name: 'Please specify Name' }).fill(longName);

                  // Verify name is entered
                  await page.getByRole('button', { name: ' Save' }).click();
                  await expect(page.locator('#name_en-error')).toContainText('Please enter no more than 255 characters');
                  await expect(page.locator('#toast-container')).toContainText('Invalid data');
            });

            test('แก้ไขชื่อ โดยแก้ไขเป็นข้อความยาวๆ - Should handle editing to long text', async ({ page }) => {
                // Create entry first
                await DetailPolicy(page);
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('ชื่อสั้น');
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');

                // Edit to long 
                const longName = 'ชื่อที่ยาวมากๆ'.repeat(50);
                await DetailPolicy(page);
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill(longName);
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Invalid data');

                await page.getByRole('textbox', { name: 'Please specify Name' }).fill(longName);
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#name_en-error')).toContainText('Please enter no more than 255 characters');
                await expect(page.locator('#toast-container')).toContainText('Invalid data');

                // Verify behavior with long text
                await expect(page.locator('#name_en-error')).toContainText('Please enter no more than 255 characters');
                await expect(page.locator('#toast-container')).toContainText('Invalid data');
            });

            //validate name field
            test('validate name field - ไม่กรอกชื่อ', async ({ page }) => {
                await DetailPolicy(page);

                // Test English name validation
                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('');
                await page.getByRole('button', { name: ' Save' }).click();

                await expect(page.locator('#name_en-error')).toContainText('Please specify Name');
                await expect(page.locator('#toast-container')).toContainText('Invalid data');

                // Switch to Thai language and test Thai name validation
                await page.getByRole('combobox', { name: 'English' }).click();
                await page.getByRole('option', { name: 'ภาษาไทย' }).locator('div').first().click();

                await page.getByRole('textbox', { name: 'Please specify Name' }).fill('');
                await page.getByRole('button', { name: ' Save' }).click();

                await expect(page.locator('#name_th-error')).toContainText('Please specify Name');
                await expect(page.locator('#toast-container')).toContainText('Invalid data');
            });
        });

        test.describe('Description Field Tests', () => {
            test('คำอธิบาย ไม่จำกัดตัวอักษร - Should allow unlimited text in description', async ({ page }) => {
                // Test empty description - should save successfully
                await DetailPolicy(page);
                await page.getByRole('textbox', { name: 'Please specify Description' }).fill('');
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');

                // Test short description - should save successfully
                await DetailPolicy(page);
                await page.getByRole('textbox', { name: 'Please specify Description' }).fill('คำอธิบายสั้น');
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');

                // Test very long description - should save successfully (unlimited)
                const longDescription = 'คำอธิบายที่ยาวมากๆ '.repeat(100);
                await DetailPolicy(page);
                await page.getByRole('textbox', { name: 'Please specify Description' }).fill(longDescription);
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');

                // Test clearing description
                await DetailPolicy(page);
                const descriptionField = page.getByRole('textbox', { name: 'Please specify Description' });
                await descriptionField.fill('');
                await page.getByRole('button', { name: ' Save' }).click();
                await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
            });

        test('แก้ไขคำอธิบาย และอักขระพิเศษ - Should allow editing description with special characters', async ({ page }) => {
            // Navigate to PDPA policy page
            await DetailPolicy(page);

            // Create entry first
            await page.getByRole('textbox', { name: 'Please specify Name' }).fill('Test PDPA');

            // Fill description field
            const descriptionField = page.getByRole('textbox', { name: 'Please specify Description' });
            await descriptionField.fill('คำอธิบายเดิม');

            await page.getByRole('button', { name: ' Save' }).click();
            await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');

            // Edit description with special characters
            await descriptionField.clear();
            const specialDescription = 'คำอธิบายใหม่ !@#$%^&*()_+{}[]|;:,.<>?';
            await descriptionField.fill(specialDescription);

            // Save changes
            await page.getByRole('button', { name: ' Save' }).click();
            await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');

            // Verify the description was updated
            await expect(descriptionField).toHaveValue(specialDescription);
        });

        test('ลบคำอธิบาย - Should allow clearing description', async ({ page }) => {
            await DetailPolicy(page);

            const descriptionField = page.getByRole('textbox', { name: 'Please specify Description' });
            await descriptionField.fill('');
            await page.getByRole('button', { name: ' Save' }).click();

            await expect(descriptionField).toHaveValue('');
            await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
        });

        test('กรอกรายละเอียดด้วยตัวอักษร - Should accept text input in description', async ({ page }) => {
            await DetailPolicy(page);
            const descriptionField = page.getByRole('textbox', { name: 'Please specify Description' });
            const textDescription = 'นี่คือรายละเอียดของนโยบาย PDPA ที่ใช้ในการทดสอบระบบ';
            await descriptionField.fill(textDescription);
            await page.getByRole('button', { name: ' Save' }).click();
            await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
            await expect(descriptionField).toContainText(textDescription);
        });
    });
    // test.describe('Rich Text Editor Tests', () => {
    //     test('เพิ่ม Image - Should allow adding images (TinyMCE)', async ({ page }) => {
    //         await DetailPolicy(page);
    //         await page.waitForLoadState('networkidle');

    //         // เปิด Insert Image
    //         await page.getByRole('button', { name: 'Insert/edit image' }).click();
    //         await page.getByRole('button', { name: 'Source' }).click();

    //         // switch ไปยัง iframe ของ elFinder
    //         const frame = await page
    //             .getByRole('dialog', { name: 'File Management' })
    //             .locator('iframe')
    //             .contentFrame();

    //         // รอให้ไฟล์ปรากฏ (เอาไฟล์แรกก็พอ)
    //         const file = frame.locator('.elfinder-cwd-file-wrapper').first();
    //         await expect(file).toBeVisible();

    //         // เลือกไฟล์
    //         await file.click();

    //         // กด OK / Insert
    //         await page.getByRole('button', { name: 'Source' }).click();

    //         // ปิด dialog
    //         await page.getByRole('dialog', { name: 'File Management' }).getByLabel('Close').click();

    //         // Save
    //         await page.getByRole('button', { name: ' Save' }).click();

    //         // ตรวจสอบผลลัพธ์
    //         await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
    //         await expect(page.locator('.ql-editor img')).toBeVisible();
    //     });





    //     // test('เพิ่ม Link บน text ที่เลือก - Should allow adding links to selected text', async ({ page }) => {
    //     //     await DetailPolicy(page);
    //     //     const descriptionField = page.getByRole('textbox', { name: 'Please specify Details' });
    //     //     await descriptionField.fill('Test with Link');
    //     //     await page.getByRole('button', { name: ' Save' }).click();
    //     //     await expect(page.locator('#toast-container')).toContainText('Successfully saved changes');
    //     //     await page.click('button:has-text("Add New")');
    //     //     await page.fill('input[name="name"]', 'Test with Link');
    //     //     await expect(linkButton).toBeVisible();
    //     //     await linkButton.click();
    //     //     await expect(page.locator('input[placeholder*="URL"], input[placeholder*="Link"]')).toBeVisible();
    //     // } else {
    //     //     await expect(linkButton).toBeVisible();
    //     // }

    //     //     const editor = page.locator('.ql-editor');
    //     //     await editor.fill('ข้อความที่จะใส่ลิงก์');

    //     //     // Select text and add link
    //     //     await editor.selectText();
    //     //     const linkButton = page.locator('.ql-toolbar button[data-value="link"], .ql-toolbar .ql-link');
    //     //     if (await linkButton.count() > 0) {
    //     //         await linkButton.click();
    //     //         // Test link dialog
    //     //         await expect(page.locator('input[placeholder*="URL"], input[placeholder*="Link"]')).toBeVisible();
    //     //     }
    //     // });

    //     //     test('เพิ่ม Media - Should allow adding media', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test with Media');

    //     //         // Look for media button
    //     //         const mediaButton = page.locator('.ql-toolbar button[data-value="video"], .ql-toolbar .ql-video');
    //     //         if (await mediaButton.count() > 0) {
    //     //             await mediaButton.click();
    //     //             await expect(page.locator('input[placeholder*="video"], input[placeholder*="URL"]')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('Insert template - Should allow inserting templates', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Template');

    //     //         // Look for template button
    //     //         const templateButton = page.locator('.ql-toolbar button[data-value="template"], .ql-toolbar .ql-template');
    //     //         if (await templateButton.count() > 0) {
    //     //             await templateButton.click();
    //     //             // Test template selection
    //     //             await expect(page.locator('.template-selector, .ql-template-dropdown')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('เพิ่ม Table - Should allow adding tables', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test with Table');

    //     //         const tableButton = page.locator('.ql-toolbar button[data-value="table"], .ql-toolbar .ql-table');
    //     //         if (await tableButton.count() > 0) {
    //     //             await tableButton.click();
    //     //             // Test table insertion
    //     //             await expect(page.locator('table')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('Special character - Should allow inserting special characters', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Special Characters');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with special chars: ');

    //     //         // Test special characters input
    //     //         await editor.type('!@#$%^&*()_+-=[]{}|;:,.<>?');

    //     //         await expect(editor).toContainText('!@#$%^&*()_+-=[]{}|;:,.<>?');
    //     //     });

    //     //     test('Emoticons - Should allow inserting emoticons', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Emoticons');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with emoticons: ');

    //     //         // Test emoticon insertion
    //     //         await editor.type('😀😊😍👍❤️🎉');

    //     //         await expect(editor).toContainText('😀😊😍👍❤️🎉');
    //     //     });

    //     //     test('เพิ่มเส้นบรรทัด - Should allow adding horizontal line', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Horizontal Line');

    //     //         const hrButton = page.locator('.ql-toolbar button[data-value="hr"], .ql-toolbar .ql-hr');
    //     //         if (await hrButton.count() > 0) {
    //     //             await hrButton.click();
    //     //             await expect(page.locator('hr')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ระบุตำแหน่ง Anchor - Should allow adding anchors', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Anchor');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with anchor');

    //     //         // Test anchor functionality
    //     //         await editor.selectText();
    //     //         const anchorButton = page.locator('.ql-toolbar button[data-value="anchor"], .ql-toolbar .ql-anchor');
    //     //         if (await anchorButton.count() > 0) {
    //     //             await anchorButton.click();
    //     //             await expect(page.locator('input[placeholder*="anchor"], input[placeholder*="name"]')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('Date/time - Should allow inserting date/time', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Date/Time');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Current date: ');

    //     //         // Test date insertion
    //     //         const dateButton = page.locator('.ql-toolbar button[data-value="date"], .ql-toolbar .ql-date');
    //     //         if (await dateButton.count() > 0) {
    //     //             await dateButton.click();
    //     //             await expect(page.locator('.ql-date-picker, .date-picker')).toBeVisible();
    //     //         }
    //     //     });
    //     // });

    //     // test.describe('Text Formatting Tests', () => {
    //     //     test('แก้ไข font เป็นตัวหนา - Should allow bold formatting', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Bold');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text to make bold');
    //     //         await editor.selectText();

    //     //         const boldButton = page.locator('.ql-toolbar button[data-value="bold"], .ql-toolbar .ql-bold');
    //     //         if (await boldButton.count() > 0) {
    //     //             await boldButton.click();
    //     //             await expect(page.locator('strong, b')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('แก้ไข font เป็นตัวเอียง - Should allow italic formatting', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Italic');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text to make italic');
    //     //         await editor.selectText();

    //     //         const italicButton = page.locator('.ql-toolbar button[data-value="italic"], .ql-toolbar .ql-italic');
    //     //         if (await italicButton.count() > 0) {
    //     //             await italicButton.click();
    //     //             await expect(page.locator('em, i')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('เพิ่มเส้นใต้ font - Should allow underline formatting', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Underline');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text to underline');
    //     //         await editor.selectText();

    //     //         const underlineButton = page.locator('.ql-toolbar button[data-value="underline"], .ql-toolbar .ql-underline');
    //     //         if (await underlineButton.count() > 0) {
    //     //             await underlineButton.click();
    //     //             await expect(page.locator('u')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ขีดทับ font - Should allow strikethrough formatting', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Strikethrough');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text to strikethrough');
    //     //         await editor.selectText();

    //     //         const strikeButton = page.locator('.ql-toolbar button[data-value="strike"], .ql-toolbar .ql-strike');
    //     //         if (await strikeButton.count() > 0) {
    //     //             await strikeButton.click();
    //     //             await expect(page.locator('s, strike, del')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ดรรชนีบน หรือ ตัวบน - Should allow superscript', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Superscript');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with superscript');
    //     //         await editor.selectText();

    //     //         const supButton = page.locator('.ql-toolbar button[data-value="super"], .ql-toolbar .ql-script[data-value="super"]');
    //     //         if (await supButton.count() > 0) {
    //     //             await supButton.click();
    //     //             await expect(page.locator('sup')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ดรรชนีบน หรือ ตัวล่าง - Should allow subscript', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Subscript');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with subscript');
    //     //         await editor.selectText();

    //     //         const subButton = page.locator('.ql-toolbar button[data-value="sub"], .ql-toolbar .ql-script[data-value="sub"]');
    //     //         if (await subButton.count() > 0) {
    //     //             await subButton.click();
    //     //             await expect(page.locator('sub')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ให้ font เป็นรูปแบบ Code - Should allow code formatting', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Code');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Code text');
    //     //         await editor.selectText();

    //     //         const codeButton = page.locator('.ql-toolbar button[data-value="code"], .ql-toolbar .ql-code');
    //     //         if (await codeButton.count() > 0) {
    //     //             await codeButton.click();
    //     //             await expect(page.locator('code')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('รูปแบบตัวอักษรต่างๆ - Should allow different font families', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Font Family');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with different font');
    //     //         await editor.selectText();

    //     //         const fontSelect = page.locator('.ql-toolbar select.ql-font');
    //     //         if (await fontSelect.count() > 0) {
    //     //             await fontSelect.selectOption('serif');
    //     //             await expect(page.locator('.ql-font-serif')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('เลือก Blocks template - Should allow block templates', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Block Template');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Select this text');
    //     //         await editor.selectText();

    //     //         const blockSelect = page.locator('.ql-toolbar select.ql-header');
    //     //         if (await blockSelect.count() > 0) {
    //     //             await blockSelect.selectOption('1');
    //     //             await expect(page.locator('h1')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('รูปแบบ font ต่างๆ - Should allow different font styles', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Font Styles');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text for styling');

    //     //         // Test multiple font styles
    //     //         await editor.selectText();

    //     //         // Test bold
    //     //         const boldButton = page.locator('.ql-toolbar .ql-bold');
    //     //         if (await boldButton.count() > 0) {
    //     //             await boldButton.click();
    //     //         }

    //     //         // Test italic
    //     //         const italicButton = page.locator('.ql-toolbar .ql-italic');
    //     //         if (await italicButton.count() > 0) {
    //     //             await italicButton.click();
    //     //         }

    //     //         await expect(editor).toContainText('Text for styling');
    //     //     });

    //     //     test('ขนาดของ font - Should allow font size changes', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Font Size');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text for size change');
    //     //         await editor.selectText();

    //     //         const sizeSelect = page.locator('.ql-toolbar select.ql-size');
    //     //         if (await sizeSelect.count() > 0) {
    //     //             await sizeSelect.selectOption('large');
    //     //             await expect(page.locator('.ql-size-large')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('จัดตำแหน่งชิดซ้าย ชิดขวา ตรงกลาง หรือปรับให้เหมาะสม - Should allow text alignment', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Text Alignment');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text to align');

    //     //         // Test center alignment
    //     //         const centerButton = page.locator('.ql-toolbar button[data-value="center"], .ql-toolbar .ql-align[data-value="center"]');
    //     //         if (await centerButton.count() > 0) {
    //     //             await centerButton.click();
    //     //             await expect(page.locator('.ql-align-center')).toBeVisible();
    //     //         }

    //     //         // Test right alignment
    //     //         const rightButton = page.locator('.ql-toolbar button[data-value="right"], .ql-toolbar .ql-align[data-value="right"]');
    //     //         if (await rightButton.count() > 0) {
    //     //             await rightButton.click();
    //     //             await expect(page.locator('.ql-align-right')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ปรับขนาดความสูงของเส้น - Should allow line height adjustment', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Line Height');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with line height');

    //     //         const lineHeightSelect = page.locator('.ql-toolbar select.ql-lineheight');
    //     //         if (await lineHeightSelect.count() > 0) {
    //     //             await lineHeightSelect.selectOption('2');
    //     //             await expect(page.locator('.ql-lineheight-2')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ปรับสีของ Text - Should allow text color changes', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Text Color');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Colored text');
    //     //         await editor.selectText();

    //     //         const colorButton = page.locator('.ql-toolbar button.ql-color, .ql-toolbar .ql-color');
    //     //         if (await colorButton.count() > 0) {
    //     //             await colorButton.click();
    //     //             await expect(page.locator('.ql-color-picker, .color-picker')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('ปรับพื้นหลังของ Text - Should allow background color changes', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Background Color');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Text with background');
    //     //         await editor.selectText();

    //     //         const bgColorButton = page.locator('.ql-toolbar button.ql-background, .ql-toolbar .ql-background');
    //     //         if (await bgColorButton.count() > 0) {
    //     //             await bgColorButton.click();
    //     //             await expect(page.locator('.ql-background-picker, .background-picker')).toBeVisible();
    //     //         }
    //     //     });

    //     //     test('Clear formatting - Should allow clearing all formatting', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Clear Formatting');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('Formatted text');
    //     //         await editor.selectText();

    //     //         // Apply some formatting first
    //     //         const boldButton = page.locator('.ql-toolbar .ql-bold');
    //     //         if (await boldButton.count() > 0) {
    //     //             await boldButton.click();
    //     //         }

    //     //         // Clear formatting
    //     //         const clearButton = page.locator('.ql-toolbar button.ql-clean, .ql-toolbar .ql-clean');
    //     //         if (await clearButton.count() > 0) {
    //     //             await clearButton.click();
    //     //             await expect(page.locator('strong, b')).not.toBeVisible();
    //     //         }
    //     //     });

    //     //     test('การแก้ไข Source code - Should allow editing source code', async ({ page }) => {
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Source Code');

    //     //         const editor = page.locator('.ql-editor');
    //     //         await editor.fill('<p>HTML content</p>');

    //     //         const sourceButton = page.locator('.ql-toolbar button.ql-source, .ql-toolbar .ql-source');
    //     //         if (await sourceButton.count() > 0) {
    //     //             await sourceButton.click();
    //     //             await expect(page.locator('textarea.ql-source, .ql-source-editor')).toBeVisible();
    //     //         }
    //     //     });
    //     // });

    //     // test.describe('Data Verification Tests', () => {
    //     //     test('เข้าตรวจสอบข้อมูลฟอร์ม แก้ไขไม่ได้ - Should allow viewing form data in read-only mode', async ({ page }) => {
    //     //         // Create a PDPA entry first
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Read-Only View');
    //     //         await page.fill('textarea[name="description"], .ql-editor', 'Test description');
    //     //         await page.click('button:has-text("Save")');

    //     //         // Navigate to view mode
    //     //         await page.click('tbody tr:first-child td:last-child a:has-text("View")');

    //     //         // Verify fields are read-only
    //     //         await expect(page.locator('input[name="name"]')).toBeDisabled();
    //     //         await expect(page.locator('textarea[name="description"], .ql-editor')).toBeDisabled();
    //     //     });

    //     //     test('สามารถดู IP Address / Browser / Timestamp ได้ - Should display tracking information', async ({ page }) => {
    //     //         // Create and save a PDPA entry
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Test Tracking Info');
    //     //         await page.click('button:has-text("Save")');

    //     //         // Navigate to view mode
    //     //         await page.click('tbody tr:first-child td:last-child a:has-text("View")');

    //     //         // Look for tracking information
    //     //         await expect(page.locator('text=IP Address, text=Browser, text=Timestamp, text=Created At, text=Updated At')).toBeVisible();
    //     //     });

    //     //     test('Check ซ้ำ IP / User แล้วปรับ Status ล่าสุด - Should handle duplicate IP/User and update latest status', async ({ page }) => {
    //     //         // Create first entry
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'First Entry');
    //     //         await page.click('button:has-text("Save")');

    //     //         // Create second entry with same user/IP
    //     //         await page.click('button:has-text("Add New")');
    //     //         await page.fill('input[name="name"]', 'Second Entry');
    //     //         await page.click('button:has-text("Save")');

    //     //         // Verify both entries exist and check for duplicate handling
    //     //         const entries = page.locator('tbody tr');
    //     //         await expect(entries).toHaveCount(2);

    //     //         // Check for status updates or duplicate warnings
    //     //         await expect(page.locator('text=Duplicate, text=Status, text=Latest')).toBeVisible();
    //     //     });
    // });
});
