import { test, expect } from '@playwright/test';

test.describe('Content Module - Complete Test Suite', () => {
  test.beforeEach(async ({ page }) => {
    // Set longer timeout for all tests
    test.setTimeout(120000);
    
    try {
      // Navigate to the admin content page
      await page.goto('http://127.0.0.1:8000/admin/products/content', { 
        timeout: 120000,
        waitUntil: 'domcontentloaded'
      });
      
      // Wait for page to load
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      // Check for login form and login if needed
      const loginForm = page.locator('form:has(input[name="username"])');
      if (await loginForm.isVisible()) {
        console.log('🔐 Login form detected, attempting to login...');
        
        // Use working credentials
        await page.fill('input[name="username"]', 'admin');
        await page.fill('input[name="password"]', 'admin');
        await page.click('button[type="submit"], input[type="submit"]');
        
        // Wait for redirect
        await page.waitForLoadState('networkidle', { timeout: 60000 });
        await page.waitForTimeout(3000);
        
        console.log('✅ Login attempted');
      } else {
        console.log('✅ No login form - may already be logged in');
      }
    } catch (error) {
      console.error('❌ Navigation failed:', error.message);
      throw error;
    }
  });

  // ===== BASIC FUNCTIONALITY TESTS =====

  test('should load the products content page successfully', async ({ page }) => {
    // Check if page loads correctly
    await expect(page).toHaveTitle('Z.com');
    
    // Check if main content is visible
    const body = page.locator('body').first();
    await expect(body).toBeVisible();
    
    // Take screenshot for debugging
    await page.screenshot({ path: 'content-products-page.png', fullPage: true });
    
    console.log('✅ Products content page loaded successfully');
  });

  test('should test page elements and navigation', async ({ page }) => {
    // Check for various navigation elements
    const elements = [
      'text=Products',
      'text=Content',
      'text=Admin',
      'text=Dashboard',
      'text=Add',
      'text=Edit',
      'text=Delete',
      'text=Search',
      'text=Save',
      'text=Cancel',
      'text=Manage',
      'text=Settings',
      'text=Profile',
      'text=User',
      'text=System'
    ];
    
    let foundElements = 0;
    for (const selector of elements) {
      const element = page.locator(selector).first();
      if (await element.isVisible()) {
        console.log(`✅ Found: ${selector}`);
        foundElements++;
      }
    }
    
    console.log(`📋 Found ${foundElements} elements out of ${elements.length}`);
    
    // Check for any content-related content
    const contentContent = page.locator('text=Product, text=Content, text=product, text=content').first();
    if (await contentContent.isVisible()) {
      console.log('✅ Products content found');
    } else {
      console.log('❌ Products content not found');
    }
  });

  // ===== PRODUCTS CONTENT CRUD TESTS =====

  test('should test products content CRUD operations', async ({ page }) => {
    console.log('🚀 Testing Products Content CRUD Operations');
    
    // Step 1: Test Add functionality
    const addButton = page.locator('text=Add, a:has-text("Add"), button:has-text("Add")').first();
    if (await addButton.isVisible()) {
      console.log('➕ Add button found, testing navigation');
      await addButton.click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      console.log(`🔗 Current URL: ${currentUrl}`);
      
      if (currentUrl.includes('/add')) {
        console.log('✅ Successfully navigated to add page');
        
        // Test form fields
        const nameInput = page.locator('input[placeholder*="Name"], input[name*="name"]').first();
        if (await nameInput.isVisible()) {
          console.log('📝 Testing name input');
          await nameInput.fill('Test Product Content');
          await page.waitForTimeout(1000);
        }
        
        // Test description field
        const descInput = page.locator('textarea[placeholder*="Description"], textarea[name*="desc"]').first();
        if (await descInput.isVisible()) {
          console.log('📝 Testing description input');
          await descInput.fill('This is a test product description for automation testing.');
          await page.waitForTimeout(1000);
        }
        
        // Test detail/content field (text editor)
        const detailInput = page.locator('textarea[name*="detail"], .ck-editor .ck-content').first();
        if (await detailInput.isVisible()) {
          console.log('📝 Testing detail/content input');
          await detailInput.click();
          await detailInput.fill('Detailed content for the test product. This includes comprehensive information about the product features and specifications.');
          await page.waitForTimeout(1000);
        }
        
        // Test image upload
        const imageUpload = page.locator('input[type="file"][accept*="image"]').first();
        if (await imageUpload.isVisible()) {
          console.log('🖼️ Testing image upload');
          // Note: In a real test, you would upload an actual image file
          // await imageUpload.setInputFiles('path/to/test/image.jpg');
        }
        
        // Test save button
        const saveButton = page.locator('button:has-text("Save"), input[type="submit"]').first();
        if (await saveButton.isVisible()) {
          console.log('💾 Testing save button');
          // Note: In a real scenario, we might want to actually save the content
          // await saveButton.click();
          // await page.waitForLoadState('networkidle', { timeout: 60000 });
        }
        
        // Test cancel/back
        const cancelButton = page.locator('text=Cancel, a:has-text("Cancel"), button:has-text("Cancel")').first();
        if (await cancelButton.isVisible()) {
          console.log('❌ Testing cancel button');
          await cancelButton.click();
          await page.waitForLoadState('networkidle', { timeout: 60000 });
          console.log('✅ Cancel button tested');
        } else {
          await page.goBack();
          console.log('⬅️ Used browser back button');
        }
      }
    }
    
    console.log('✅ Products Content CRUD operations tested');
  });

  // ===== NEWS CONTENT TESTS =====

  test('should test news content functionality', async ({ page }) => {
    console.log('📰 Testing News Content Functionality');
    
    // Navigate to news content
    await page.goto('http://127.0.0.1:8000/admin/news/content', { 
      timeout: 120000,
      waitUntil: 'domcontentloaded'
    });
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    // Test page load
    const body = page.locator('body').first();
    await expect(body).toBeVisible();
    console.log('✅ News content page loaded');
    
    // Test table functionality
    const mainTable = page.locator('table:not(.phpdebugbar-widgets-params):not(.phpdebugbar-widgets-tablevar)').first();
    if (await mainTable.isVisible()) {
      console.log('✅ News content table found');
      
      // Check table headers
      const headers = mainTable.locator('th');
      const headerCount = await headers.count();
      console.log(`📊 News table has ${headerCount} headers`);
      
      // Test search functionality
      const searchInput = page.locator('input[placeholder*="Search"], input[type="search"]').first();
      if (await searchInput.isVisible()) {
        await searchInput.fill('Test News');
        await page.waitForTimeout(2000);
        await searchInput.fill('');
        console.log('🔍 News search functionality tested');
      }
    }
    
    console.log('✅ News content functionality tested');
  });

  // ===== ABOUT PAGE TESTS =====

  test('should test about page functionality', async ({ page }) => {
    console.log('ℹ️ Testing About Page Functionality');
    
    // Navigate to about page
    await page.goto('http://127.0.0.1:8000/admin/about/page', { 
      timeout: 120000,
      waitUntil: 'domcontentloaded'
    });
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    // Test page load
    const body = page.locator('body').first();
    await expect(body).toBeVisible();
    console.log('✅ About page loaded');
    
    // Test form fields if available
    const nameInput = page.locator('input[name*="name"], input[placeholder*="Name"]').first();
    if (await nameInput.isVisible()) {
      await nameInput.fill('About Us Test Content');
      await page.waitForTimeout(1000);
      console.log('📝 About page name field tested');
    }
    
    // Test text editor if available
    const detailInput = page.locator('textarea[name*="detail"], .ck-editor .ck-content').first();
    if (await detailInput.isVisible()) {
      await detailInput.click();
      await detailInput.fill('This is test content for the about us page. It includes company information and history.');
      await page.waitForTimeout(1000);
      console.log('📝 About page detail field tested');
    }
    
    console.log('✅ About page functionality tested');
  });

  // ===== SERVICE CONTENT TESTS =====

  test('should test service content functionality', async ({ page }) => {
    console.log('🔧 Testing Service Content Functionality');
    
    // Navigate to service content
    await page.goto('http://127.0.0.1:8000/admin/service/content', { 
      timeout: 120000,
      waitUntil: 'domcontentloaded'
    });
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    // Test page load
    const body = page.locator('body').first();
    await expect(body).toBeVisible();
    console.log('✅ Service content page loaded');
    
    // Test add service functionality
    const addButton = page.locator('text=Add, a:has-text("Add"), button:has-text("Add")').first();
    if (await addButton.isVisible()) {
      await addButton.click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      if (currentUrl.includes('/add')) {
        console.log('✅ Successfully navigated to add service page');
        
        // Test service form fields
        const serviceNameInput = page.locator('input[name*="name"], input[placeholder*="Name"]').first();
        if (await serviceNameInput.isVisible()) {
          await serviceNameInput.fill('Test Service');
          console.log('📝 Service name field tested');
        }
        
        const serviceDescInput = page.locator('textarea[name*="desc"], textarea[placeholder*="Description"]').first();
        if (await serviceDescInput.isVisible()) {
          await serviceDescInput.fill('Test service description for automation testing.');
          console.log('📝 Service description field tested');
        }
        
        // Go back
        const cancelButton = page.locator('text=Cancel, a:has-text("Cancel"), button:has-text("Cancel")').first();
        if (await cancelButton.isVisible()) {
          await cancelButton.click();
        } else {
          await page.goBack();
        }
        await page.waitForLoadState('networkidle', { timeout: 60000 });
      }
    }
    
    console.log('✅ Service content functionality tested');
  });

  // ===== FILE UPLOAD TESTS =====

  test('should test file upload functionality', async ({ page }) => {
    console.log('📁 Testing File Upload Functionality');
    
    // Navigate to products content add page
    await page.goto('http://127.0.0.1:8000/admin/products/content/add', { 
      timeout: 120000,
      waitUntil: 'domcontentloaded'
    });
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    // Test image upload fields
    const imageUploads = page.locator('input[type="file"][accept*="image"]');
    const imageUploadCount = await imageUploads.count();
    console.log(`🖼️ Found ${imageUploadCount} image upload fields`);
    
    if (imageUploadCount > 0) {
      // Test first image upload field
      const firstImageUpload = imageUploads.first();
      if (await firstImageUpload.isVisible()) {
        console.log('📤 Testing image upload field');
        // Note: In a real test environment, you would upload actual files
        // await firstImageUpload.setInputFiles('path/to/test/image.jpg');
      }
    }
    
    // Test video upload fields
    const videoUploads = page.locator('input[type="file"][accept*="video"]');
    const videoUploadCount = await videoUploads.count();
    console.log(`🎥 Found ${videoUploadCount} video upload fields`);
    
    // Test file upload fields
    const fileUploads = page.locator('input[type="file"]:not([accept*="image"]):not([accept*="video"])');
    const fileUploadCount = await fileUploads.count();
    console.log(`📄 Found ${fileUploadCount} general file upload fields`);
    
    console.log('✅ File upload functionality tested');
  });

  // ===== SEARCH AND FILTER TESTS =====

  test('should test search and filter functionality', async ({ page }) => {
    console.log('🔍 Testing Search and Filter Functionality');
    
    // Test search functionality
    const searchInputs = page.locator('input[placeholder*="Search"], input[type="search"], input[placeholder*="search"]');
    const searchCount = await searchInputs.count();
    console.log(`🔍 Found ${searchCount} search inputs`);
    
    if (searchCount > 0) {
      const searchInput = searchInputs.first();
      if (await searchInput.isVisible()) {
        console.log('✅ Search input found and visible');
        
        // Test search functionality
        await searchInput.fill('Test Product');
        await page.waitForTimeout(2000);
        
        // Test search with special characters
        await searchInput.fill('Test @ # $ %');
        await page.waitForTimeout(2000);
        
        // Clear search
        await searchInput.fill('');
        await page.waitForTimeout(2000);
        
        console.log('✅ Search functionality tested');
      }
    }
    
    // Test filter functionality
    const filterSelects = page.locator('select[name*="filter"], select[name*="category"]');
    const filterCount = await filterSelects.count();
    console.log(`🎛️ Found ${filterCount} filter selects`);
    
    if (filterCount > 0) {
      const filterSelect = filterSelects.first();
      if (await filterSelect.isVisible()) {
        console.log('✅ Filter select found');
        
        // Test filter options
        const options = filterSelect.locator('option');
        const optionCount = await options.count();
        console.log(`📋 Filter has ${optionCount} options`);
        
        if (optionCount > 1) {
          // Select second option (first is usually "All" or default)
          await filterSelect.selectOption({ index: 1 });
          await page.waitForTimeout(2000);
          
          // Reset to first option
          await filterSelect.selectOption({ index: 0 });
          await page.waitForTimeout(2000);
          
          console.log('✅ Filter functionality tested');
        }
      }
    }
    
    console.log('✅ Search and filter functionality tested');
  });

  // ===== MULTI-LANGUAGE TESTS =====

  test('should test multi-language functionality', async ({ page }) => {
    console.log('🌐 Testing Multi-language Functionality');
    
    // Navigate to products content add page
    await page.goto('http://127.0.0.1:8000/admin/products/content/add', { 
      timeout: 120000,
      waitUntil: 'domcontentloaded'
    });
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    // Look for language selector
    const langSelectors = page.locator('select[name*="lang"], .language-selector, button:has-text("English"), button:has-text("ภาษาไทย")');
    const langCount = await langSelectors.count();
    console.log(`🌐 Found ${langCount} language selectors`);
    
    if (langCount > 0) {
      const langSelector = langSelectors.first();
      if (await langSelector.isVisible()) {
        console.log('✅ Language selector found');
        
        // Test switching languages
        if (await langSelector.evaluate(el => el.tagName) === 'SELECT') {
          // If it's a select element
          const options = langSelector.locator('option');
          const optionCount = await options.count();
          console.log(`🌐 Found ${optionCount} language options`);
          
          if (optionCount > 1) {
            // Switch to second language
            await langSelector.selectOption({ index: 1 });
            await page.waitForTimeout(2000);
            
            // Switch back to first language
            await langSelector.selectOption({ index: 0 });
            await page.waitForTimeout(2000);
            
            console.log('✅ Language switching tested');
          }
        } else {
          // If it's a button or other element
          await langSelector.click();
          await page.waitForTimeout(2000);
          console.log('✅ Language selector clicked');
        }
      }
    }
    
    // Test multilingual form fields
    const nameFields = page.locator('input[name*="name"]');
    const nameFieldCount = await nameFields.count();
    console.log(`📝 Found ${nameFieldCount} name fields (may include different languages)`);
    
    if (nameFieldCount > 0) {
      for (let i = 0; i < Math.min(nameFieldCount, 2); i++) {
        const nameField = nameFields.nth(i);
        if (await nameField.isVisible()) {
          await nameField.fill(`Test Product Name ${i + 1}`);
          await page.waitForTimeout(500);
        }
      }
      console.log('✅ Multi-language name fields tested');
    }
    
    console.log('✅ Multi-language functionality tested');
  });

  // ===== TABLE FUNCTIONALITY TESTS =====

  test('should test table functionality with sorting and pagination', async ({ page }) => {
    console.log('📊 Testing Table Functionality');
    
    // Look for main data table
    const mainTable = page.locator('table:not(.phpdebugbar-widgets-params):not(.phpdebugbar-widgets-tablevar)').first();
    if (await mainTable.isVisible()) {
      console.log('✅ Main data table found');
      
      // Test table headers
      const headers = mainTable.locator('th');
      const headerCount = await headers.count();
      console.log(`📊 Table has ${headerCount} headers`);
      
      if (headerCount > 0) {
        for (let i = 0; i < Math.min(headerCount, 5); i++) {
          const header = headers.nth(i);
          const text = await header.textContent();
          console.log(`  Header ${i}: "${text}"`);
        }
      }
      
      // Test sorting functionality
      const sortableHeaders = mainTable.locator('th[class*="sorting"], th[data-orderable="true"]');
      const sortableCount = await sortableHeaders.count();
      console.log(`🔄 Found ${sortableCount} sortable headers`);
      
      if (sortableCount > 0) {
        const firstSortableHeader = sortableHeaders.first();
        if (await firstSortableHeader.isVisible()) {
          await firstSortableHeader.click();
          await page.waitForTimeout(2000);
          
          // Click again to reverse sort
          await firstSortableHeader.click();
          await page.waitForTimeout(2000);
          
          console.log('✅ Table sorting tested');
        }
      }
      
      // Test table rows
      const rows = mainTable.locator('tbody tr');
      const rowCount = await rows.count();
      console.log(`📊 Table has ${rowCount} data rows`);
      
      if (rowCount > 0) {
        // Test first row content
        const firstRow = rows.first();
        const cells = firstRow.locator('td');
        const cellCount = await cells.count();
        console.log(`📊 First row has ${cellCount} cells`);
        
        // Test action buttons in first row
        const editButton = firstRow.locator('a[href*="/edit/"], button:has-text("Edit")');
        const deleteButton = firstRow.locator('button[title="Delete"], button:has-text("Delete")');
        
        if (await editButton.isVisible()) {
          console.log('✏️ Edit button found in table row');
        }
        
        if (await deleteButton.isVisible()) {
          console.log('🗑️ Delete button found in table row');
        }
      }
      
      // Test pagination
      const paginationElements = page.locator('.pagination, .dataTables_paginate, button:has-text("Previous"), button:has-text("Next")');
      const paginationCount = await paginationElements.count();
      console.log(`📄 Found ${paginationCount} pagination elements`);
      
      if (paginationCount > 0) {
        const nextButton = page.locator('button:has-text("Next"), a:has-text("Next")').first();
        if (await nextButton.isVisible() && !await nextButton.isDisabled()) {
          await nextButton.click();
          await page.waitForTimeout(2000);
          console.log('✅ Pagination next button tested');
          
          // Go back to previous page
          const prevButton = page.locator('button:has-text("Previous"), a:has-text("Previous")').first();
          if (await prevButton.isVisible() && !await prevButton.isDisabled()) {
            await prevButton.click();
            await page.waitForTimeout(2000);
            console.log('✅ Pagination previous button tested');
          }
        }
      }
      
    } else {
      console.log('❌ Main data table not found');
    }
    
    console.log('✅ Table functionality tested');
  });

  // ===== DELETE FUNCTIONALITY TESTS =====

  test('should test delete functionality with confirmation', async ({ page }) => {
    console.log('🗑️ Testing Delete Functionality');
    
    // Look for delete buttons
    const deleteButtons = page.locator('button[title="Delete"], button:has-text("Delete"), .delete-btn');
    const deleteCount = await deleteButtons.count();
    
    if (deleteCount > 0) {
      console.log(`🗑️ Found ${deleteCount} delete buttons`);
      
      // Click first delete button
      await deleteButtons.first().click();
      await page.waitForTimeout(2000);
      
      // Look for confirmation dialog
      const dialog = page.locator('dialog, .modal, [role="dialog"], .swal2-container').first();
      if (await dialog.isVisible()) {
        console.log('💬 Delete confirmation dialog found');
        
        // Check dialog content
        const dialogText = await dialog.textContent();
        console.log(`💬 Dialog content: ${dialogText}`);
        
        // Look for cancel button
        const cancelBtn = page.locator('button:has-text("Cancel"), button:has-text("No"), button:has-text("ยกเลิก")').first();
        if (await cancelBtn.isVisible()) {
          console.log('❌ Testing cancel delete');
          await cancelBtn.click();
          await page.waitForTimeout(1000);
          console.log('✅ Delete cancelled successfully');
        } else {
          console.log('❌ No cancel button found in dialog');
          await page.keyboard.press('Escape');
        }
      } else {
        console.log('❌ No delete confirmation dialog found');
      }
    } else {
      console.log('❌ No delete buttons found');
    }
    
    console.log('✅ Delete functionality tested');
  });

  // ===== RESPONSIVE DESIGN TESTS =====

  test('should test responsive design across different viewports', async ({ page }) => {
    console.log('📱 Testing Responsive Design');
    
    // Test different viewport sizes
    const viewports = [
      { width: 1920, height: 1080, name: 'Desktop Large' },
      { width: 1024, height: 768, name: 'Desktop Small' },
      { width: 768, height: 1024, name: 'Tablet' },
      { width: 375, height: 667, name: 'Mobile' }
    ];
    
    for (const viewport of viewports) {
      try {
        await page.setViewportSize(viewport);
        await page.waitForTimeout(2000);
        
        // Wait for page to stabilize after viewport change
        await page.waitForLoadState('domcontentloaded', { timeout: 10000 });
        
        // Check if main content is still visible
        const mainContent = page.locator('body').first();
        if (await mainContent.isVisible()) {
          console.log(`✅ Responsive design test passed for ${viewport.name} (${viewport.width}x${viewport.height})`);
        } else {
          console.log(`⚠️ Main content not visible for ${viewport.name}, but continuing...`);
        }
        
        // Test navigation menu responsiveness
        const navMenu = page.locator('nav, .navbar, .sidebar, .menu').first();
        if (await navMenu.isVisible()) {
          console.log(`📱 Navigation menu visible on ${viewport.name}`);
        }
        
        // Test table responsiveness on smaller screens
        if (viewport.width <= 768) {
          const table = page.locator('table').first();
          if (await table.isVisible()) {
            console.log(`📊 Table responsive on ${viewport.name}`);
          }
        }
        
      } catch (error) {
        console.log(`❌ Responsive design test failed for ${viewport.name}: ${error.message}`);
      }
    }
    
    console.log('✅ Responsive design testing completed');
  });

  // ===== ACCESSIBILITY TESTS =====

  test('should test accessibility features', async ({ page }) => {
    console.log('♿ Testing Accessibility Features');
    
    // Check for proper heading structure
    const headings = page.locator('h1, h2, h3, h4, h5, h6');
    const headingCount = await headings.count();
    console.log(`📝 Found ${headingCount} headings on the page`);
    
    if (headingCount > 0) {
      for (let i = 0; i < Math.min(headingCount, 5); i++) {
        const heading = headings.nth(i);
        const text = await heading.textContent();
        const tagName = await heading.evaluate(el => el.tagName);
        console.log(`  ${tagName}: "${text}"`);
      }
    }
    
    // Check for form labels
    const labels = page.locator('label');
    const labelCount = await labels.count();
    console.log(`🏷️ Found ${labelCount} form labels`);
    
    // Check for images with alt text
    const images = page.locator('img');
    const imageCount = await images.count();
    console.log(`🖼️ Found ${imageCount} images on the page`);
    
    if (imageCount > 0) {
      let imagesWithAlt = 0;
      for (let i = 0; i < Math.min(imageCount, 5); i++) {
        const image = images.nth(i);
        const alt = await image.getAttribute('alt');
        const src = await image.getAttribute('src');
        if (alt && alt.trim() !== '') {
          imagesWithAlt++;
        }
        console.log(`  Image ${i}: alt="${alt}", src="${src}"`);
      }
      console.log(`✅ ${imagesWithAlt} images have alt text out of ${Math.min(imageCount, 5)} checked`);
    }
    
    // Check for proper form structure
    const forms = page.locator('form');
    const formCount = await forms.count();
    console.log(`📋 Found ${formCount} forms on the page`);
    
    if (formCount > 0) {
      const form = forms.first();
      const inputs = form.locator('input, select, textarea');
      const inputCount = await inputs.count();
      console.log(`📝 First form has ${inputCount} input elements`);
      
      // Check for required field indicators
      const requiredInputs = form.locator('input[required], select[required], textarea[required]');
      const requiredCount = await requiredInputs.count();
      console.log(`⚠️ Found ${requiredCount} required fields`);
    }
    
    // Test keyboard navigation
    console.log('⌨️ Testing keyboard navigation');
    await page.keyboard.press('Tab');
    await page.waitForTimeout(500);
    await page.keyboard.press('Tab');
    await page.waitForTimeout(500);
    await page.keyboard.press('Tab');
    console.log('✅ Keyboard navigation tested');
    
    console.log('✅ Accessibility features tested');
  });

  // ===== PERFORMANCE TESTS =====

  test('should test performance and error handling', async ({ page }) => {
    console.log('⚡ Testing Performance and Error Handling');
    
    // Measure page load time
    const startTime = Date.now();
    await page.waitForLoadState('networkidle');
    const endTime = Date.now();
    const loadTime = endTime - startTime;
    
    console.log(`⏱️ Page load time: ${loadTime}ms`);
    
    // Check if load time is reasonable (less than 10 seconds)
    expect(loadTime).toBeLessThan(10000);
    
    // Check for error messages
    const errorMessages = page.locator('text=error, text=Error, .error, .alert-danger, .alert-error');
    const errorCount = await errorMessages.count();
    console.log(`❌ Found ${errorCount} error messages on the page`);
    
    // Check for success messages
    const successMessages = page.locator('text=success, text=Success, .success, .alert-success');
    const successCount = await successMessages.count();
    console.log(`✅ Found ${successCount} success messages on the page`);
    
    // Check for warning messages
    const warningMessages = page.locator('text=warning, text=Warning, .warning, .alert-warning');
    const warningCount = await warningMessages.count();
    console.log(`⚠️ Found ${warningCount} warning messages on the page`);
    
    // Test with invalid input to check error handling
    const searchInput = page.locator('input[placeholder*="Search"]').first();
    if (await searchInput.isVisible()) {
      await searchInput.fill('<script>alert("XSS Test")</script>');
      await page.waitForTimeout(2000);
      
      // Should handle gracefully without executing script
      const alertDialog = page.locator('[role="alert"], .alert');
      const alertCount = await alertDialog.count();
      if (alertCount === 0) {
        console.log('✅ XSS input handled safely');
      }
      
      // Clear the input
      await searchInput.fill('');
      await page.waitForTimeout(1000);
    }
    
    console.log('✅ Performance and error handling tested');
  });

  // ===== COMPLETE WORKFLOW INTEGRATION TEST =====

  test('Complete Content Module Workflow Test', async ({ page }) => {
    console.log('🚀 Starting Complete Content Module Workflow Test');
    
    // Step 1: Verify page loads
    await expect(page).toHaveTitle('Z.com');
    console.log('✅ Step 1: Page loaded successfully');
    
    // Step 2: Test Products Content
    console.log('📦 Step 2: Testing Products Content');
    await page.goto('http://127.0.0.1:8000/admin/products/content');
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    const productsTable = page.locator('table').first();
    if (await productsTable.isVisible()) {
      console.log('✅ Products table loaded');
    }
    
    // Step 3: Test News Content
    console.log('📰 Step 3: Testing News Content');
    await page.goto('http://127.0.0.1:8000/admin/news/content');
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    const newsTable = page.locator('table').first();
    if (await newsTable.isVisible()) {
      console.log('✅ News table loaded');
    }
    
    // Step 4: Test About Page
    console.log('ℹ️ Step 4: Testing About Page');
    await page.goto('http://127.0.0.1:8000/admin/about/page');
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    const aboutForm = page.locator('form').first();
    if (await aboutForm.isVisible()) {
      console.log('✅ About page form loaded');
    }
    
    // Step 5: Test Service Content
    console.log('🔧 Step 5: Testing Service Content');
    await page.goto('http://127.0.0.1:8000/admin/service/content');
    await page.waitForLoadState('networkidle', { timeout: 60000 });
    
    const serviceTable = page.locator('table').first();
    if (await serviceTable.isVisible()) {
      console.log('✅ Service table loaded');
    }
    
    // Step 6: Take final screenshot
    await page.screenshot({ path: 'content-module-workflow-complete.png', fullPage: true });
    console.log('✅ Step 6: Final screenshot taken');
    
    console.log('🎉 Complete Content Module Workflow Test finished successfully!');
  });
});
