import { test, expect } from '@playwright/test';

test.describe('Contact Subject Module - Complete Test Suite', () => {
  test.beforeEach(async ({ page }) => {
    // Set longer timeout for all tests
    test.setTimeout(120000);
    
    try {
      // Navigate to the contact subject page
      await page.goto('http://127.0.0.1:8000/admin/contact/subject', { 
        timeout: 120000,
        waitUntil: 'domcontentloaded'
      });
      
      // Wait for page to load
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      // Check for login form and login if needed
      const loginForm = page.locator('form:has(input[name="username"])');
      if (await loginForm.isVisible()) {
        console.log('🔐 Login form detected, attempting to login...');
        
        // Use working credentials
        await page.fill('input[name="username"]', 'admin');
        await page.fill('input[name="password"]', 'admin');
        await page.click('button[type="submit"], input[type="submit"]');
        
        // Wait for redirect
        await page.waitForLoadState('networkidle', { timeout: 60000 });
        await page.waitForTimeout(3000);
        
        console.log('✅ Login attempted');
      } else {
        console.log('✅ No login form - may already be logged in');
      }
    } catch (error) {
      console.error('❌ Navigation failed:', error.message);
      throw error;
    }
  });

  // ===== BASIC FUNCTIONALITY TESTS =====

  test('should load the contact subject page successfully', async ({ page }) => {
    // Check if page loads correctly
    await expect(page).toHaveTitle('Z.com');
    
    // Check if main content is visible
    const body = page.locator('body').first();
    await expect(body).toBeVisible();
    
    // Take screenshot for debugging
    await page.screenshot({ path: 'contact-subject-page.png', fullPage: true });
    
    console.log('✅ Contact subject page loaded successfully');
  });

  test('should test page elements and navigation', async ({ page }) => {
    // Check for various navigation elements
    const elements = [
      'text=Contact',
      'text=Subject',
      'text=Admin',
      'text=Dashboard',
      'text=Add',
      'text=Edit',
      'text=Delete',
      'text=Search',
      'text=Save',
      'text=Cancel',
      'text=Manage',
      'text=Settings',
      'text=Profile',
      'text=User',
      'text=System'
    ];
    
    let foundElements = 0;
    for (const selector of elements) {
      const element = page.locator(selector).first();
      if (await element.isVisible()) {
        console.log(`✅ Found: ${selector}`);
        foundElements++;
      }
    }
    
    console.log(`📋 Found ${foundElements} elements out of ${elements.length}`);
    
    // Check for any contact-related content
    const contactContent = page.locator('text=Contact, text=Subject, text=contact, text=subject').first();
    if (await contactContent.isVisible()) {
      console.log('✅ Contact subject content found');
    } else {
      console.log('❌ Contact subject content not found');
    }
  });

  // ===== FORM FUNCTIONALITY TESTS =====

  test('should test add subject functionality', async ({ page }) => {
    // Look for Add button
    const addButton = page.locator('text=Add, a:has-text("Add"), button:has-text("Add")').first();
    if (await addButton.isVisible()) {
      console.log('➕ Add button found, testing navigation');
      await addButton.click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      console.log(`🔗 Current URL: ${currentUrl}`);
      
      if (currentUrl.includes('/add')) {
        console.log('✅ Successfully navigated to add page');
        
        // Test form fields
        const nameInput = page.locator('input[placeholder*="Name"], input[name*="name"]').first();
        if (await nameInput.isVisible()) {
          console.log('📝 Testing name input');
          await nameInput.fill('Test Subject');
          await page.waitForTimeout(1000);
          await nameInput.fill('');
        }
        
        // Test description input
        const descInput = page.locator('input[placeholder*="Description"], textarea[name*="description"]').first();
        if (await descInput.isVisible()) {
          console.log('📝 Testing description input');
          await descInput.fill('Test Description');
          await page.waitForTimeout(1000);
          await descInput.fill('');
        }
        
        // Test status select
        const statusSelect = page.locator('select[name*="status"], select[name*="Status"]').first();
        if (await statusSelect.isVisible()) {
          console.log('📝 Testing status select');
          const options = await statusSelect.locator('option').count();
          console.log(`Status select has ${options} options`);
        }
        
        // Test cancel/back
        const cancelButton = page.locator('text=Cancel, a:has-text("Cancel"), button:has-text("Cancel")').first();
        if (await cancelButton.isVisible()) {
          console.log('❌ Testing cancel button');
          await cancelButton.click();
          await page.waitForLoadState('networkidle', { timeout: 60000 });
          console.log('✅ Cancel button tested');
        } else {
          await page.goBack();
          console.log('⬅️ Used browser back button');
        }
      } else {
        console.log('❌ Did not navigate to add page');
      }
    } else {
      console.log('❌ Add button not found');
    }
  });

  test('should test edit subject functionality', async ({ page }) => {
    // Look for edit links in table
    const editLinks = page.locator('a[href*="/edit/"], button:has-text("Edit"), .edit-btn');
    const editCount = await editLinks.count();
    
    if (editCount > 0) {
      console.log(`✏️ Found ${editCount} edit links`);
      
      // Click first edit link
      await editLinks.first().click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      console.log(`🔗 Current URL: ${currentUrl}`);
      
      if (currentUrl.includes('/edit/')) {
        console.log('✅ Successfully navigated to edit page');
        
        // Test form fields
        const nameInput = page.locator('input[placeholder*="Name"], input[name*="name"]').first();
        if (await nameInput.isVisible()) {
          console.log('📝 Testing name input in edit mode');
          const currentValue = await nameInput.inputValue();
          console.log(`Current name value: ${currentValue}`);
          await nameInput.fill('Updated Subject Name');
          await page.waitForTimeout(1000);
          await nameInput.fill(currentValue); // Restore original value
        }
        
        // Test cancel/back
        const cancelButton = page.locator('text=Cancel, a:has-text("Cancel"), button:has-text("Cancel")').first();
        if (await cancelButton.isVisible()) {
          console.log('❌ Testing cancel button in edit mode');
          await cancelButton.click();
          await page.waitForLoadState('networkidle', { timeout: 60000 });
          console.log('✅ Cancel button tested in edit mode');
        } else {
          await page.goBack();
          console.log('⬅️ Used browser back button from edit page');
        }
      } else {
        console.log('❌ Did not navigate to edit page');
      }
    } else {
      console.log('❌ No edit links found');
    }
  });

  test('should test view subject functionality', async ({ page }) => {
    // Look for view links in table
    const viewLinks = page.locator('a[href*="/view/"], button:has-text("View"), .view-btn');
    const viewCount = await viewLinks.count();
    
    if (viewCount > 0) {
      console.log(`👁️ Found ${viewCount} view links`);
      
      // Click first view link
      await viewLinks.first().click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      console.log(`🔗 Current URL: ${currentUrl}`);
      
      if (currentUrl.includes('/view/')) {
        console.log('✅ Successfully navigated to view page');
        
        // Check if form is in read-only mode
        const nameInput = page.locator('input[placeholder*="Name"], input[name*="name"]').first();
        if (await nameInput.isVisible()) {
          const isDisabled = await nameInput.isDisabled();
          console.log(`Name input is ${isDisabled ? 'disabled (read-only)' : 'enabled'}`);
        }
        
        // Go back
        const backButton = page.locator('text=Back, a:has-text("Back"), button:has-text("Back")').first();
        if (await backButton.isVisible()) {
          await backButton.click();
        } else {
          await page.goBack();
        }
        await page.waitForLoadState('networkidle', { timeout: 60000 });
        console.log('✅ View functionality tested');
      } else {
        console.log('❌ Did not navigate to view page');
      }
    } else {
      console.log('❌ No view links found');
    }
  });

  // ===== SEARCH FUNCTIONALITY TESTS =====

  test('should test search functionality', async ({ page }) => {
    // Look for search input
    const searchInputs = page.locator('input[placeholder*="Search"], input[type="search"], input[placeholder*="search"]');
    const searchCount = await searchInputs.count();
    console.log(`🔍 Found ${searchCount} search inputs`);
    
    if (searchCount > 0) {
      const searchInput = searchInputs.first();
      if (await searchInput.isVisible()) {
        console.log('✅ Search input found and visible');
        
        // Test search functionality
        await searchInput.fill('Test Subject');
        await page.waitForTimeout(2000);
        
        // Clear search
        await searchInput.fill('');
        await page.waitForTimeout(2000);
        
        console.log('✅ Search functionality tested');
      } else {
        console.log('❌ Search input found but not visible');
      }
    } else {
      console.log('❌ Search input not found');
    }
  });

  // ===== TABLE FUNCTIONALITY TESTS =====

  test('should test table functionality', async ({ page }) => {
    // Look for tables on the page
    const tables = page.locator('table');
    const tableCount = await tables.count();
    console.log(`📊 Found ${tableCount} tables on the page`);
    
    if (tableCount > 0) {
      // Find main table (exclude debug tables)
      const mainTable = page.locator('table:not(.phpdebugbar-widgets-params):not(.phpdebugbar-widgets-tablevar)').first();
      if (await mainTable.isVisible()) {
        console.log('✅ Main table found');
        
        // Check for table headers
        const headers = mainTable.locator('th');
        const headerCount = await headers.count();
        console.log(`📊 Table has ${headerCount} headers`);
        
        if (headerCount > 0) {
          for (let i = 0; i < Math.min(headerCount, 5); i++) {
            const header = headers.nth(i);
            const text = await header.textContent();
            console.log(`  Header ${i}: "${text}"`);
          }
        }
        
        // Check for table rows
        const rows = mainTable.locator('tbody tr');
        const rowCount = await rows.count();
        console.log(`📊 Table has ${rowCount} data rows`);
        
        if (rowCount > 0) {
          // Log first row content
          const firstRow = rows.first();
          const cells = firstRow.locator('td');
          const cellCount = await cells.count();
          console.log(`📊 First row has ${cellCount} cells`);
          
          for (let i = 0; i < Math.min(cellCount, 3); i++) {
            const cell = cells.nth(i);
            const text = await cell.textContent();
            console.log(`  Cell ${i}: "${text}"`);
          }
        }
      } else {
        console.log('❌ Main table not found');
      }
    }
  });

  // ===== DELETE FUNCTIONALITY TESTS =====

  test('should test delete functionality', async ({ page }) => {
    // Look for delete buttons
    const deleteButtons = page.locator('button[title="Delete"], button:has-text("Delete"), .delete-btn');
    const deleteCount = await deleteButtons.count();
    
    if (deleteCount > 0) {
      console.log(`🗑️ Found ${deleteCount} delete buttons`);
      
      // Click first delete button
      await deleteButtons.first().click();
      await page.waitForTimeout(2000);
      
      // Look for confirmation dialog
      const dialog = page.locator('dialog, .modal, [role="dialog"]').first();
      if (await dialog.isVisible()) {
        console.log('💬 Delete confirmation dialog found');
        
        // Look for cancel button
        const cancelBtn = page.locator('button:has-text("Cancel"), button:has-text("No")').first();
        if (await cancelBtn.isVisible()) {
          console.log('❌ Testing cancel delete');
          await cancelBtn.click();
          await page.waitForTimeout(1000);
          console.log('✅ Delete cancelled successfully');
        } else {
          console.log('❌ No cancel button found in dialog');
          await page.keyboard.press('Escape');
        }
      } else {
        console.log('❌ No delete confirmation dialog found');
      }
    } else {
      console.log('❌ No delete buttons found');
    }
  });

  // ===== RESPONSIVE DESIGN TESTS =====

  test('should test responsive design', async ({ page }) => {
    // Test different viewport sizes
    const viewports = [
      { width: 1024, height: 768 },
      { width: 768, height: 1024 },
      { width: 375, height: 667 }
    ];
    
    for (const viewport of viewports) {
      try {
        await page.setViewportSize(viewport);
        await page.waitForTimeout(2000);
        
        // Wait for page to stabilize after viewport change
        await page.waitForLoadState('domcontentloaded', { timeout: 10000 });
        
        // Check if main content is still visible
        const mainContent = page.locator('body').first();
        if (await mainContent.isVisible()) {
          console.log(`✅ Responsive design test passed for ${viewport.width}x${viewport.height}`);
        } else {
          console.log(`⚠️ Main content not visible for ${viewport.width}x${viewport.height}, but continuing...`);
        }
      } catch (error) {
        console.log(`❌ Responsive design test failed for ${viewport.width}x${viewport.height}: ${error.message}`);
        // Continue with next viewport instead of failing the entire test
      }
    }
  });

  // ===== ACCESSIBILITY TESTS =====

  test('should test accessibility features', async ({ page }) => {
    // Check for proper heading structure
    const headings = page.locator('h1, h2, h3, h4, h5, h6');
    const headingCount = await headings.count();
    console.log(`📝 Found ${headingCount} headings on the page`);
    
    if (headingCount > 0) {
      for (let i = 0; i < Math.min(headingCount, 3); i++) {
        const heading = headings.nth(i);
        const text = await heading.textContent();
        const tagName = await heading.evaluate(el => el.tagName);
        console.log(`  ${tagName}: "${text}"`);
      }
    }
    
    // Check for form labels
    const labels = page.locator('label');
    const labelCount = await labels.count();
    console.log(`🏷️ Found ${labelCount} form labels`);
    
    // Check for images with alt text
    const images = page.locator('img');
    const imageCount = await images.count();
    console.log(`🖼️ Found ${imageCount} images on the page`);
    
    if (imageCount > 0) {
      for (let i = 0; i < Math.min(imageCount, 3); i++) {
        const image = images.nth(i);
        const alt = await image.getAttribute('alt');
        const src = await image.getAttribute('src');
        console.log(`  Image ${i}: alt="${alt}", src="${src}"`);
      }
    }
  });

  // ===== COMPLETE WORKFLOW TESTS =====

  test('Complete workflow: Add, Edit, View, Search, Delete', async ({ page }) => {
    console.log('🚀 Starting complete workflow test for Contact Subject');
    
    // Step 1: Verify page loads
    await expect(page).toHaveTitle('Z.com');
    console.log('✅ Step 1: Page loaded');
    
    // Step 2: Test Add functionality
    const addButton = page.locator('text=Add, a:has-text("Add"), button:has-text("Add")').first();
    if (await addButton.isVisible()) {
      console.log('✅ Step 2: Add button found');
      await addButton.click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      if (currentUrl.includes('/add')) {
        console.log('✅ Step 2: Successfully navigated to add page');
        
        // Test form fields
        const nameInput = page.locator('input[placeholder*="Name"], input[name*="name"]').first();
        if (await nameInput.isVisible()) {
          await nameInput.fill('Test Subject Workflow');
          console.log('✅ Step 2: Form field tested');
        }
        
        // Go back
        const cancelButton = page.locator('text=Cancel, a:has-text("Cancel"), button:has-text("Cancel")').first();
        if (await cancelButton.isVisible()) {
          await cancelButton.click();
        } else {
          await page.goBack();
        }
        await page.waitForLoadState('networkidle', { timeout: 60000 });
      }
    }
    
    // Step 3: Test Search functionality
    const searchInput = page.locator('input[placeholder*="Search"], input[type="search"]').first();
    if (await searchInput.isVisible()) {
      console.log('✅ Step 3: Search input found');
      await searchInput.fill('Test');
      await page.waitForTimeout(2000);
      await searchInput.fill('');
      console.log('✅ Step 3: Search functionality tested');
    }
    
    // Step 4: Test Edit functionality
    const editLinks = page.locator('a[href*="/edit/"], button:has-text("Edit")');
    const editCount = await editLinks.count();
    if (editCount > 0) {
      console.log('✅ Step 4: Edit links found');
      await editLinks.first().click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      if (currentUrl.includes('/edit/')) {
        console.log('✅ Step 4: Successfully navigated to edit page');
        
        // Go back
        const cancelButton = page.locator('text=Cancel, a:has-text("Cancel"), button:has-text("Cancel")').first();
        if (await cancelButton.isVisible()) {
          await cancelButton.click();
        } else {
          await page.goBack();
        }
        await page.waitForLoadState('networkidle', { timeout: 60000 });
      }
    }
    
    // Step 5: Test View functionality
    const viewLinks = page.locator('a[href*="/view/"], button:has-text("View")');
    const viewCount = await viewLinks.count();
    if (viewCount > 0) {
      console.log('✅ Step 5: View links found');
      await viewLinks.first().click();
      await page.waitForLoadState('networkidle', { timeout: 60000 });
      
      const currentUrl = page.url();
      if (currentUrl.includes('/view/')) {
        console.log('✅ Step 5: Successfully navigated to view page');
        
        // Go back
        const backButton = page.locator('text=Back, a:has-text("Back"), button:has-text("Back")').first();
        if (await backButton.isVisible()) {
          await backButton.click();
        } else {
          await page.goBack();
        }
        await page.waitForLoadState('networkidle', { timeout: 60000 });
      }
    }
    
    // Step 6: Test Delete functionality
    const deleteButtons = page.locator('button[title="Delete"], button:has-text("Delete")');
    const deleteCount = await deleteButtons.count();
    if (deleteCount > 0) {
      console.log('✅ Step 6: Delete buttons found');
      await deleteButtons.first().click();
      await page.waitForTimeout(2000);
      
      // Cancel delete
      const cancelBtn = page.locator('button:has-text("Cancel"), button:has-text("No")').first();
      if (await cancelBtn.isVisible()) {
        await cancelBtn.click();
        console.log('✅ Step 6: Delete cancelled successfully');
      } else {
        await page.keyboard.press('Escape');
      }
    }
    
    // Step 7: Take final screenshot
    await page.screenshot({ path: 'contact-subject-workflow-complete.png', fullPage: true });
    console.log('✅ Step 7: Final screenshot taken');
    
    console.log('🎉 Complete workflow test finished successfully!');
  });

  // ===== ERROR HANDLING AND PERFORMANCE TESTS =====

  test('should test error handling and performance', async ({ page }) => {
    // Check for error messages
    const errorMessages = page.locator('text=error, text=Error, .error, .alert-danger, .alert-error');
    const errorCount = await errorMessages.count();
    console.log(`❌ Found ${errorCount} error messages on the page`);
    
    // Check for success messages
    const successMessages = page.locator('text=success, text=Success, .success, .alert-success');
    const successCount = await successMessages.count();
    console.log(`✅ Found ${successCount} success messages on the page`);
    
    // Check for warning messages
    const warningMessages = page.locator('text=warning, text=Warning, .warning, .alert-warning');
    const warningCount = await warningMessages.count();
    console.log(`⚠️ Found ${warningCount} warning messages on the page`);
    
    // Measure page performance
    const startTime = Date.now();
    await page.waitForLoadState('networkidle');
    const endTime = Date.now();
    const loadTime = endTime - startTime;
    
    console.log(`⏱️ Page load time: ${loadTime}ms`);
    
    // Check if load time is reasonable (less than 10 seconds)
    expect(loadTime).toBeLessThan(10000);
  });
});
