<?php

namespace App\Imports;

use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithStartRow;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Modules\Member\app\Models\Member;
use Modules\Member\app\Models\MemberLavel;

class MemberImport implements ToCollection, WithStartRow, SkipsOnError, SkipsOnFailure
{
    use SkipsErrors, SkipsFailures;

    private $customErrors = [];
    private $successCount = 0;
    private $errorCount = 0;

    /**
     * Start reading from row 2 (skip header row)
     */
    public function startRow(): int
    {
        return 2;
    }

    /**
     * @param Collection $rows
     */
    public function collection(Collection $rows)
    {
        foreach ($rows as $index => $row) {
            try {
                // Convert row to array if it's a Collection
                $rowArray = $row instanceof \Illuminate\Support\Collection ? $row->toArray() : (array) $row;
                
                // Get column values using numeric index (0-based)
                // Column order: รหัสพนักงาน, ชื่อ, อีเมล, แผนก, ตำแหน่ง, ระดับสมาชิก, ผู้อนุมัติ 1, ผู้อนุมัติ 2
                $code = $rowArray[0] ?? null;
                $name = $rowArray[1] ?? null;
                $email = $rowArray[2] ?? null;
                $department = $rowArray[3] ?? null;
                $position = $rowArray[4] ?? null;
                $lavelName = $rowArray[5] ?? null;
                $approveFirst = $rowArray[6] ?? null;
                $approveSecond = $rowArray[7] ?? null;
                
                // Ignore row if both name and email are empty
                if (empty($name) && empty($email)) {
                    continue;
                }

                // Check required 'name'
                if (empty($name)) {
                    $this->errorCount++;
                    $this->customErrors[] = [
                        'row' => $index + 2,
                        'message' => 'ชื่อเป็นข้อมูลที่จำเป็น'
                    ];
                    continue;
                }

                // Check required 'email'
                if (empty($email)) {
                    $this->errorCount++;
                    $this->customErrors[] = [
                        'row' => $index + 2,
                        'message' => 'อีเมลเป็นข้อมูลที่จำเป็น'
                    ];
                    continue;
                }

                // Lookup lavel by name if provided
                $lavel_id = null;
                if (!empty($lavelName)) {
                    $lavel = MemberLavel::where('name', 'like', '%' . trim($lavelName) . '%')->first();
                    if (!$lavel) {
                        $this->errorCount++;
                        $this->customErrors[] = [
                            'row' => $index + 2,
                            'message' => 'Lavel ไม่พบ'
                        ];
                    } else {
                        $lavel_id = $lavel->id;
                    }
                }

                // Helper to resolve Member by name for approval fields
                $findMemberIdByName = function ($nameInput, $label) use ($index) {
                    if (!empty($nameInput)) {
                        $member = Member::where('name', 'like', '%' . trim($nameInput) . '%')->first();
                        if ($member) {
                            return $member->id;
                        } else {
                            $this->errorCount++;
                            $this->customErrors[] = [
                                'row' => $index + 2,
                                'message' => $label . ' ไม่พบ'
                            ];
                        }
                    }
                    return null;
                };

                $approve_by_first_id = $findMemberIdByName($approveFirst, 'Approve by first');
                $approve_by_second_id = $findMemberIdByName($approveSecond, 'Approve by second');

                // Build member data array
                $data = [
                    'type' => 'member',
                    'code' => !empty($code) ? trim($code) : null,
                    'name' => trim($name),
                    'email' => !empty($email) ? trim($email) : null,
                    'department' => !empty($department) ? trim($department) : null,
                    'position' => !empty($position) ? trim($position) : null,
                    'approve_by_first_id' => $approve_by_first_id,
                    'approve_by_second_id' => $approve_by_second_id,
                    'lavel_id' => $lavel_id,
                    'status' => 0,
                ];

                // Basic email format check
                if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
                    $this->errorCount++;
                    $this->customErrors[] = [
                        'row' => $index + 2,
                        'message' => 'รูปแบบอีเมลไม่ถูกต้อง'
                    ];
                    continue;
                }

                // Create or update member by unique email (and type)
                $member = null;
                if (!empty($data['email'])) {
                    $member = Member::where('email', $data['email'])
                        ->where('type', 'member')
                        ->first();
                    if ($member) {
                        $member->update($data);
                    } else {
                        $member = Member::create($data);
                    }
                } else {
                    $member = Member::create($data);
                }

                // Handle parent_id relationship logic (if needed, using $row[8] for parent, but not required by prompt)
                // Optionally ignore since prompt only requests [0]..[7]

                $this->successCount++;
            } catch (\Exception $e) {
                $this->errorCount++;
                $this->customErrors[] = [
                    'row' => $index + 2,
                    'message' => $e->getMessage()
                ];
            }
        }
    }


    /**
     * Get import errors
     */
    public function getErrors()
    {
        // Combine validation errors and custom errors
        $allErrors = $this->customErrors;
        
        foreach ($this->failures() as $failure) {
            $allErrors[] = [
                'row' => $failure->row(),
                'message' => implode(', ', $failure->errors())
            ];
        }

        return $allErrors;
    }

    /**
     * Get success count
     */
    public function getSuccessCount()
    {
        return $this->successCount;
    }

    /**
     * Get error count
     */
    public function getErrorCount()
    {
        return $this->errorCount + count($this->failures());
    }
}

