<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Symfony\Component\HttpFoundation\Response as BaseResponse;

class CustomValidatePostSize
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $max = $this->getPostMaxSize();

        if ($max > 0 && $request->server('CONTENT_LENGTH') > $max) {
            // Check if this is an AJAX request
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'error' => true,
                    'msg' => 'File size exceeds the limit',
                    'max_size' => $max,
                    'max_size_formatted' => $this->formatBytes($max)
                ], 413); // 413 Payload Too Large
            }

            // For regular requests, redirect with error message
            return redirect()->back()->withErrors([
                'file_size' => 'File size exceeds the limit'
            ]);
        }

        return $next($request);
    }

    /**
     * Get the maximum POST size in bytes.
     * Set to 2MB as requested
     *
     * @return int
     */
    protected function getPostMaxSize()
    {
        // Set maximum POST size to 2MB
        return 2 * 1024 * 1024; // 2MB in bytes
        
        // Alternative: Read from config file
        // return config('upload.max_post_size', '2M');
    }

    /**
     * Format bytes to human readable format.
     *
     * @param  int  $bytes
     * @param  int  $precision
     * @return string
     */
    protected function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, $precision) . ' ' . $units[$i];
    }
}
