<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Modules\Map\app\Models\CheckInOut;
use Carbon\Carbon;

class CleanupCheckInOutData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'checkinout:cleanup {--months=3 : จำนวนเดือนย้อนหลังที่ต้องการเก็บไว้}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'ลบข้อมูลการเดินทาง (check_in_out) ที่เก่ากว่าจำนวนเดือนที่กำหนด (ค่าเริ่มต้น 3 เดือน)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $months = $this->option('months');
        
        // ตรวจสอบว่าเป็นตัวเลขที่ถูกต้อง
        if (!is_numeric($months) || $months < 1) {
            $this->error('จำนวนเดือนต้องเป็นตัวเลขที่มากกว่า 0');
            return Command::FAILURE;
        }

        // คำนวณวันที่ย้อนหลัง
        $thresholdDate = Carbon::now()->subMonths($months)->startOfDay();
        
        $this->info("กำลังลบข้อมูลการเดินทางที่เก่ากว่า {$months} เดือน (ก่อนวันที่ {$thresholdDate->format('d/m/Y')})...");
        
        // นับจำนวนข้อมูลที่จะลบ
        $count = CheckInOut::where('created_at', '<', $thresholdDate)->count();
        
        if ($count === 0) {
            $this->info('ไม่พบข้อมูลที่ต้องลบ');
            return Command::SUCCESS;
        }
        
        // ยืนยันการลบ
        if ($this->confirm("พบข้อมูล {$count} รายการที่จะลบ คุณต้องการดำเนินการต่อหรือไม่?", true)) {
            try {
                // ลบข้อมูล
                $deleted = CheckInOut::where('created_at', '<', $thresholdDate)->delete();
                
                $this->info("✓ ลบข้อมูลสำเร็จ จำนวน {$deleted} รายการ");
                $this->info("วันที่อ้างอิง: " . $thresholdDate->format('d/m/Y H:i:s'));
                
                // Log การลบข้อมูล
                \Log::info("CleanupCheckInOutData: ลบข้อมูล check_in_out จำนวน {$deleted} รายการ (เก่ากว่า {$months} เดือน, ก่อนวันที่ {$thresholdDate->format('Y-m-d')})");
                
                return Command::SUCCESS;
            } catch (\Exception $e) {
                $this->error("เกิดข้อผิดพลาด: " . $e->getMessage());
                \Log::error("CleanupCheckInOutData Error: " . $e->getMessage());
                return Command::FAILURE;
            }
        } else {
            $this->info('ยกเลิกการลบข้อมูล');
            return Command::SUCCESS;
        }
    }
}

