<?php

namespace Modules\OilPrice\app\Http\Controllers;

use App\Exports\OilPriceExport;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Member\app\Models\MemberLavel;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\OilPrice\app\Models\OilPrice;
use Modules\OilPrice\app\Models\OilpriceAbc;

class OilPriceAdminController extends AdminController
{
    private $module, $method, $action, $config;

    /**
     * Function : construct
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    public function __construct()
    {
        $routeName = Route::currentRouteName();
        if ($routeName) {
            $explode = explode('.', $routeName);
            $this->module = $explode[1] ?? '';
            $this->method = $explode[2] ?? '';
            $this->action = $explode[3] ?? '';
        } else {
            $this->module = '';
            $this->method = '';
            $this->action = '';
        }

        self::$navbar = [
            ['name' => __("oilprice::oilprice.{$this->method}.title"), 'url' => null],
            ['name' => __("oilprice::oilprice.{$this->method}.name"), 'url' => route_has("admin.{$this->module}.{$this->method}.index")]
        ];
        $this->config = config("oilprice.oilprice.{$this->method}");
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $limit = !empty($this->config['setting']['limit']['status']) ? $this->config['setting']['limit']['max'] : null;
        return view('oilprice::oilprice.index', [
            'navbar' => self::$navbar,
            'config' => $this->config,
            'method' => $this->method,
            'type' => $this->module,
            'limit' => $limit
        ]);
    }

    /**
     * Function : form
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    public function form($id = 0)
    {
        $data = null;
        if (!empty($id)) {
            $data = OilPrice::find($id);
            $oilprice_abc_checked = OilpriceAbc::where('role_id', Auth::user()->role_id)
                ->where('status', 1)
                ->pluck('name')
                ->toArray();
        }
        $gap_range = OilPrice::where('scheme_type', 'oil')->get(['gap_range', 'gap_min', 'gap_max', 'reference', 'default'])->toArray();
        return view('oilprice::oilprice.form', [
            'type' => $this->module,
            'method' => $this->method,
            'config' => $this->config,
            'navbar' => self::$navbar,
            'action' => $this->action,
            'data' => $data,
            'gap_range' => $gap_range,
            'oilprice_abc_checked' => $oilprice_abc_checked ?? [],
        ]);
    }


    /**
     * Function : datatable ajax
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['id']; // Use 'id' instead of '_lft' since we don't use nested set
            foreach ($this->config['table']['header'] as $h) {
                if (!empty($h['status']))
                    $dt_column[] = $h['column'];
            }
            if (!empty($this->config['table']['btn']['sort']))
                $dt_column[] = 'sort';
            array_push($dt_column, 'updated_at', 'action');

            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');

            // create Content object
            $req = OilPrice::where('status', 1);

            // Filter by method (oil, manager, sale)
            if ($this->method == 'oil') {
                // For 'oil' method, show all data (manager and sale)
                $req = $req->where('scheme_type', $this->method);
            } else {
                // For manager/sale, filter by scheme_type
                $req = $req->where('scheme_type', "!=", "Oil");
            }

            // search 
            if (!empty($dt_search)) {
                $req = $req->where(function ($q) use ($dt_search) {
                    $q->where('gap_range', 'like', "%{$dt_search}%")
                        ->orWhere('scheme_type', 'like', "%{$dt_search}%")
                        ->orWhere('reference', 'like', "%{$dt_search}%");
                });
            }

            // count all 
            $dt_total = $req->count();
            if ($dt_order == 0 && empty($this->config['table']['btn']['sort']))
                $req = $req->orderBy('updated_at', 'DESC');
            else
                $req = $req->orderBy($dt_column[$dt_order], $dt_dir);

            $resp =    $req->offset($dt_start)
                ->limit($dt_length)
                ->get();
            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('oilprice_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('gap_range', function ($record) {
                    return $record->gap_range ?? '-';
                })->editColumn('reference', function ($record) {
                    return $record->reference ? number_format($record->reference, 2) : '-';
                })->editColumn('scheme_type', function ($record) {
                    return "<span class='badge badge-secondary'>" . ucfirst($record->scheme_type_name->name ?? '-') . "</span>";
                })->editColumn('value_a', function ($record) {
                    return number_format($record->value_a ?? 0, 2);
                })->editColumn('value_b', function ($record) {
                    return number_format($record->value_b ?? 0, 2);
                })->editColumn('value_c', function ($record) {
                    return number_format($record->value_c ?? 0, 2);
                })->editColumn('rate', function ($record) {
                    return $record->rate ? number_format($record->rate, 2) : '-';
                })->editColumn('sort', function ($record) use ($dt_order) {
                    if (!empty($this->config['table']['btn']['sort']))
                        return self::btn_sort("admin.{$this->module}.{$this->method}.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
                })->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->addColumn('action', function ($record) {
                    $class = (!empty($record->default) and $record->default == 1) ? 'disabled' : '';
                    $btn = '<div class="button-list">';
                    if (!empty($this->config['table']['btn']['default']))
                        $btn .= self::btn_default("admin.{$this->module}.{$this->method}.set_default", $record->id, 'setDefault', $record->default, $class ?? "");
                    if (!empty($this->config['table']['btn']['status']))
                        $btn .= self::btn_status("admin.{$this->module}.{$this->method}.set_status", $record->id, 'setStatus', $record->status);
                    if (!empty($this->config['table']['btn']['edit']))
                        $btn .= self::btn_route("admin.{$this->module}.{$this->method}.edit", $record->id);
                    if (!empty($this->config['table']['btn']['delete'])) {

                        $btn .= self::btn_delete("admin.{$this->module}.{$this->method}.set_delete", $record->id, 'setDelete', $class ?? "");
                    }
                    $btn .= '</div>';
                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    public function save(Request $request)
    {
        $rule = $msg = [];
        $navbar = $this->config['form']['navbar'];
        $sidebar = $this->config['form']['sidebar'];
        $limit = !empty($this->config['setting']['limit']['status']) ? $this->config['setting']['limit']['max'] : null;
        if (!empty($limit['status']) && !empty($request->get('default')) && (OilPrice::where([['default', 1], ['id', '!=', $request->get('id')]])->count() >= $limit['max']))
            return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);

        if (!empty($navbar['translate']['status'])) {
            foreach (languages() as $lang => $l) {
                foreach ($navbar['translate']['input'] as $k => $i) {
                    if (!empty($i['status']) && !empty($i['validate'])) {
                        $rule["{$k}_{$lang}"] = ($k != 'name') ? 'required' : 'required|max:255';
                        $msg["{$k}_{$lang}.*"] = __("field.{$k}_placeholder") . ' ' . ($l['name'] ?? '');
                    }
                }
            }
        }
        foreach ($navbar as $k_n => $n) {
            if ($k_n == 'translate' || empty($n['status']) || empty($n['input']))
                continue;

            foreach ($n['input'] as $k => $i) {
                if (empty($i['status']) || empty($i['validate']))
                    continue;

                if (in_array($k_n, ['upload', 'file', 'video', 'gallery'])) {
                    if (empty($request->get('id')) || !empty($request->get($k . '_del')) || (empty($request->get($k . '_old') && empty($request->get($k)))) and !empty($i['validate']))
                        $rule[$k] = 'required';
                    else
                        $rule[$k . '_old'] = ['required'];
                } else
                    $rule[$k] = 'required';

                $msg["{$k}*.*"] = __("field.error.{$k}_placeholder");
            }
        }
        foreach ($sidebar as $s) {
            if (empty($s['status']) || empty($s['input']))
                continue;

            foreach ($s['input'] as $k => $i) {
                if (!empty($i['status']) && !empty($i['validate'])) {
                    if ($i['type'] == 'time') {
                        $rule["{$i['start']}"] = 'required';
                        $rule["{$i['end']}"] = 'required';
                        $msg["{$i['start']}.*"] = __("field.{$k}_placeholder");
                        $msg["{$i['end']}.*"] = __("field.{$k}_placeholder");
                    } else {
                        $rule["{$k}"] = 'required';
                        $msg["{$k}.*"] = __("field.{$k}_placeholder");
                    }
                }
            }
        }

        $valid = validator($request->all(), $rule, $msg);
        if ($valid->fails())
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_BAD_REQUEST);

        $attr = [
            'status' => $request->get('status') ? $request->get('status') : 1,
        ];

        // Handle all navbar input fields
        foreach ($navbar as $k_n => $n) {
            if ($k_n == 'translate' || empty($n['status']) || empty($n['input']))
                continue;

            foreach ($n['input'] as $k => $i) {
                if (!empty($i['status'])) {
                    // Check input type
                    if (in_array($i['type'], ['selection', 'text', 'number', 'date', 'datetime', 'textarea'])) {
                        $attr[$k] = $request->get($k);
                    }
                }
            }
        }
        if (!empty($navbar['scheme']['status'])) {
            foreach ($navbar['scheme']['input'] as $k_u => $u) {
                if (!empty($u['status'])) {
                    $attr[$k_u] = $request->get($k_u);
                }
            }
        }
        $attr['rate'] = $request->get('rate') ?? null;
        $attr['gap_range'] = $request->get('gap_range') ?? null;
        if ($request->filled('scheme_type')) {
            $attr['scheme_type'] = $request->get('scheme_type') ?? null;
        }
        $identifier = [];
        if ($request->filled('id')) {
            $identifier['id'] = $request->get('id');
            $identifier['scheme_type'] = !empty($request->get('scheme_type')) ? $request->get('scheme_type') : ($this->method ?? null);
        } else {
            $identifier['scheme_type'] = !empty($request->get('scheme_type')) ? $request->get('scheme_type') : ($this->method ?? null);
        }
        $set = OilPrice::updateOrCreate(
            $identifier,
            $attr
        );

        if ($set) {
            return self::response(['msg' => __('noti.success'), 'id' => $set->id, 'url' => route("admin.{$this->module}.{$this->method}.index")]);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : set translate
     * Dev : Petch
     * Update Date : 22 Jan 2025
     */
    private function save_upload($request, $name, $type)
    {
        if (check_file_upload($request, $name, $type))
            return ['status' => false, 'msg' => __('noti.mime_image')];

        return ['status' => true, 'result' => self::set_image_upload($request, $name, "public/{$this->module}", "{$type}-")];
    }

    /**
     * Function : set translate
     * Dev : Petch
     * Update Date : 22 Jan 2025
     */
    private function setLangs(Request $request, $row, $config)
    {
        foreach (array_keys(languages()) as $lang) {
            $attr = [];
            foreach ($config as $key => $item)
                $attr[$key] = !empty($request->get("{$key}_{$lang}")) ? mwz_setTextString($request->get("{$key}_{$lang}"), $item['type'] == 'texteditor') : null;

            $row->langs()->updateOrCreate(['lang' => $lang], $attr);
        }
    }

    /**
     * Function : save gallery
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function save_gallery(Request $request)
    {
        if (!empty($request->get('id'))) {
            $field_name = 'gallery';
            $upload_path = "/storage/{$this->module}/gallery";
            $images = $remove = [];
            // remove file if has
            if (!empty($request->get('id'))) {
                // delete file from multiple
                if ($request->get("{$field_name}_file_removed") != '') {
                    $remove_list = json_decode($request->get("{$field_name}_file_removed"));
                    if (!empty($remove_list)) {
                        foreach ($remove_list as $rm_file) {
                            $remove[] = $rm_file;
                            $remove_file = str_replace('/storage', '', $rm_file);
                            if (Storage::disk('public')->exists($remove_file))
                                Storage::disk('public')->delete($remove_file);
                        }
                    }
                }
            }
            // upload file multiple
            $uploaded = [];
            if ($request->hasFile($field_name)) {
                foreach ($request->file($field_name) as $file) {
                    if ($file->getClientMimeType() !== $file->getMimeType() || !in_array($file->getClientMimeType(), mime_type(['image'])))
                        continue;

                    $new_filename = $file->getClientOriginalName();
                    $file->move(public_path($upload_path), $new_filename);
                    $uploaded[] = $upload_path . '/' . $new_filename;
                }
            }

            // set image json 
            $index = 0;
            $file_lists = json_decode($request->get("{$field_name}_file_list"), 1);
            if (!empty($file_lists)) {
                foreach ($file_lists as $fl) {
                    if ($fl['complete'] == false) {
                        $image = $uploaded[$index++] ?? '';
                        if (empty($image))
                            continue;

                        $fl['image'] =  $image;
                        $fl['complete'] = true;
                        $images[] = $fl;
                    } elseif (!in_array($fl['image'], $remove))
                        $images[] = $fl;
                }
            }

            $set = OilPrice::updateOrCreate(['id' => $request->get('id')], ['gallery' => !empty($images) ? json_encode($images, JSON_UNESCAPED_UNICODE) : null]);
            if ($set->save())
                return self::response(['msg' => __('noti.success'), 'id' => $set->id,  'url' => route("admin.{$this->module}.{$this->method}.index")]);

            return self::response(['msg' => __('noti.error'), 'id' => $request->get('id')], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : set default
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_default(Request $request)
    {
        $id = $request->get('id');
        $set = OilPrice::find($id);
        if (!$set) {
            return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if (!$set->default) {
            // Set all other records' default to 0 and this one to 1
            OilPrice::where('default', 1)->update(['default' => 0]);
            $set->default = 1;
            if ($set->save()) {
                //คำนวณ rate ตาม gap_range
                $this->CalRate($set->reference);
                return self::response(['msg' => __('noti.success')]);
            } else {
                return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
            }
        } else {
            // Prevent having 0 default records
            $countDefault = OilPrice::where('default', 1)->count();

            $this->CalRate($set->reference);
            if ($countDefault <= 1) {
                return self::response(['msg' => __('noti.default.min_one')], Response::HTTP_BAD_REQUEST);
            } else {
                $set->default = 0;
                if ($set->save()) {
                    return self::response(['msg' => __('noti.success')]);
                } else {
                    return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
                }
            }
        }
    }

    /**
     * Calculate and update the rate for all OilPrice records with scheme_type not 'oil'.
     *
     * @param float $gap_range
     * @return void
     */
    public function CalRate($gap_range)
    {
        // Only recalculate for scheme_type NOT 'oil' (typo fixed from 'oli' to 'oil')
        $schemes = OilPrice::where('scheme_type', '!=', 'oil')->get();

        foreach ($schemes as $scheme) {
            // Avoid division by zero
            $valueA = $scheme->value_a ?: 1;

            // Calculate components
            $a = round($gap_range / $valueA, 2, PHP_ROUND_HALF_UP);
            $b = $scheme->value_b ?? 0;
            $c = $scheme->value_c ?? 0;

            // Calculate total rate
            $rate = round($a + $b + $c, 2, PHP_ROUND_HALF_UP);

            // Assign rate (cast as float, database should cast as decimal)
            $scheme->rate = $rate;
            $scheme->save();
        }
    }

    /**
     * Function : set status
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_status(Request $request)
    {
        $setting = $this->config['form']['sidebar']['setting'];
        $set = OilPrice::find($request->get('id'));
        $set->status = !$set->status;
        if ($set->save())
            return self::response(['msg' => __('noti.success')]);

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : delete
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_delete(Request $request)
    {
        $set = OilPrice::find($request->get('id'));
        if ($set->delete()) {
            $limit = 0;
            if (!empty($this->config['setting']['limit']['status']))
                $limit = OilPrice::whereNotNull('parent_id')->count() < $this->config['setting']['limit']['max'];

            return self::response(['msg' => __('noti.delete'), 'limit' => $limit]);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_BAD_REQUEST);
    }

    /**
     * Function : set sort
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_sort(Request $request)
    {
        $node = OilPrice::find($request->get('id'));
        $move = $request->get('move') == 'up' ? $node->up() : $node->down();
        if ($move)
            return self::response(['msg' =>  __('noti.sort')]);

        return self::response(['msg' => __('noti.sort_error')], Response::HTTP_BAD_REQUEST);
    }

    /**
     * Function : re order
     * Dev : Wan
     * Update Date : 29 Apr 2024
     */
    public function re_order()
    {
        // Not using nested set, so no need to fix tree
        return self::response(['msg' => 'Re-order is not applicable for this module']);
    }

    /**
     * Function : export to excel
     * Dev : AI Assistant
     * Update Date : 1 Oct 2025
     */
    public function export(Request $request)
    {
        $query = OilPrice::query();

        // Apply search filter if provided
        $search = $request->get('search');
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('gap_range', 'like', "%{$search}%")
                    ->orWhere('scheme_type', 'like', "%{$search}%")
                    ->orWhere('reference', 'like', "%{$search}%");
            });
        }

        // Apply scheme_type filter if provided
        $schemeType = $request->get('scheme_type');
        if (!empty($schemeType)) {
            $query->where('scheme_type', $schemeType);
        }

        $data = $query->orderBy('gap_min', 'ASC')
            ->orderBy('scheme_type', 'ASC')
            ->get();

        $filename = 'oilprice_' . date('Y-m-d_His') . '.xlsx';

        return Excel::download(new OilPriceExport($data), $filename);
    }

    /**
     * Function : get scheme type list for filter
     * Dev : Wan
     * Update Date : 1 Oct 2025
     */
    public function get_scheme_type_list(Request $request)
    {
        $req = MemberLavel::where('type', 'member')->whereNot('id', 1);
        $data = $req->get();
        $result = [];
        foreach ($data as $list)
            $result[] = ['id' => $list->id, 'name' => $list->name];
        return response()->json($result);
    }

    public function get_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = MemberLavel::where('type', 'member')->whereNot('id', 1);

        if (!empty($filter)) {
            if (!empty($filter['search']))
                $req = $req->where('name', 'like', "%{$filter['search']}%");

            if (!empty($filter['id']))
                $req = $req->where('id', '<>', $filter['id']);

            if (!empty($filter['category_id']))
                $req = $req->whereRelation('category', 'id',  $filter['category_id']);
        }

        $data = $req->get();
        $result = [];
        foreach ($data as $list)
            $result[] = ['id' => $list->id, 'text' => $list->name, 'image' => $list->image ?? ''];

        return self::response(['results' => $result]);
    }
}
