<?php

namespace Modules\Mwz\app\Http\Controllers;

use Illuminate\Http\Request;
use Modules\Mwz\app\Models\Slugs;

use Spatie\Sitemap\Sitemap;
use Spatie\Sitemap\Tags\Url;
use Yajra\DataTables\Facades\DataTables;

class SlugAdminController extends AdminController
{

	/**
	 * Function : construct
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{
		$this->middleware('auth:admin');
		self::$navbar = [
			['name' => __('mwz::slug.title'), 'url' => null],
			['name' => __('mwz::slug.name'), 'url' => route_has('admin.mwz.slug.index')]
		];
	}

	/**
	 * Function : index
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function index()
	{
		// $this->generate(); // Generate Slug default
		return view('mwz::slug.index', ['navbar' => self::$navbar]);
	}

	/**
	 * Function : datatable ajax
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {

			//init datatable
			$dt_column = array('id', 'slug', 'module', 'method', 'updated_at', 'action');
			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');
			// create brand object
			$req = new Slugs();
			$req = $req->where('lang', app()->getLocale());

			if (!empty($dt_search)) {
				$req = $req->where(function ($query) use ($dt_search) {
					$query->where('module', 'like', "%{$dt_search}%")
						->orwhere('method', 'like', "%{$dt_search}%")
						->orwhere('slug', 'like', "%{$dt_search}%");
				});
			}

			// module
			if (!empty($dt_filter['module'])) {
				$req = $req->where('module', $dt_filter['module']);
			}
			// method
			if (!empty($dt_filter['method'])) {
				$req = $req->where('method', $dt_filter['method']);
			}

			$dt_total = $req->count();
			// set query order & limit from datatable
			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)->limit($dt_length)
				->get();

			// prepare datatable for response
			$tables = DataTables::of($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('slug_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('slug', function ($record) {
					return limit($record->slug, 50);
				})->editColumn('module', function ($record) {
					return strtoupper($record->module);
				})->editColumn('method', function ($record) {
					return strtoupper($record->method);
				})->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					$btn .= self::btn_route('admin.mwz.slug.edit', $record->uid);
					if (empty($record->type) || !empty($record->type) && $record->type != 1) {
						$btn .= self::btn_delete('admin.mwz.slug.set_delete', $record->uid, 'setDelete');
					}
					$btn .= '</div>';
					return $btn;
				})
				->escapeColumns([]);

			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : form
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function form($uid)
	{
		self::$navbar[] = ['name' => __('action.add'), 'url' => null];
		$id = 0;
		$meta = [];
		if (!empty($uid)) {
			list($module, $method, $id) = explode('-', $uid);
			$meta = SlugController::getMetadata($module, $method, $id);
		}
		return view('mwz::slug.form', ['data_id' => $id,  'meta' => $meta, 'navbar' => self::$navbar]);
	}

	/**
	 * Function : save
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function save(Request $request)
	{
		if (!empty(SlugController::validatorSlugUpdate($request))) {
			return self::response(SlugController::validatorSlugUpdate($request), 303);
		}
		SlugController::createMetadata($request, $request->get('data_id'));
		return self::response(['msg' => __('noti.success')]);
	}

	/**
	 * Function : get list
	 * Dev : Petch
	 * Update Date : 15 Jan 2025
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = Slugs::where('lang', app()->getLocale());
		if (!empty($filter['search']))
			$req = $req->where('slug', 'like', "%{$filter['search']}%");

		if (!empty($filter['id']))
			$req = $req->where('id', '<>', $filter['id']);

		$req = $req->get();
		$result = [];
		foreach ($req as $item)
			$result[] = ['id' => $item->uid, 'text' => $item->slug, 'image' => ''];

		return self::response(['results' => $result]);
	}


	/**
	 * Function : delete
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function set_delete(Request $request)
	{
		$set = Slugs::where('uid', $request->get('id'))->get();
		if ($set->delete()) {
			return self::response(['msg' => __('noti.delete')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : gen sitemap
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function generate_sitemap()
	{

		$slug = Slugs::where('status', 1)->orderBy('level')->get();

		$sitemap = new Sitemap;
		if (!empty($slug)) {
			$sitemap->create();
			foreach ($slug as $value) {
				$lang = ($value->lang ?? 'th') == app()->getLocale() ? '/' : "/{$value->lang}/";
				$level = 0;
				switch ($value->level) {
					case 1:
						$level = 0.9;
						break;
					case 2:
						$level = 0.8;
						break;
					default:
						$level = 0.7;
						break;
				}
				$sitemap->add(Url::create($lang . $value->slug)->setLastModificationDate(now())
					->setChangeFrequency(Url::CHANGE_FREQUENCY_MONTHLY)->setPriority($level));
			}
			$sitemap->writeToFile(public_path('sitemap.xml'));
		}
		// return $sitemap;
		return redirect()->back();
	}


	private function generate()
	{
		Slugs::truncate();
		$slug = [
			//Home
			['type' => 1, 'level' => 1, 'uid' => 'index-index-1', 'slug' => 'หน้าแรก', 'lang' => 'th', 'module' => 'index', 'method' => 'index', 'data_id' => 1, 'status' => 1],
			['type' => 1, 'level' => 1, 'uid' => 'index-index-1', 'slug' => 'index', 'lang' => 'en', 'module' => 'index', 'method' => 'index', 'data_id' => 1, 'status' => 1],

			//recorddata
			['type' => 1, 'level' => 1, 'uid' => 'recorddata-recorddata-1', 'slug' => 'บันทึกข้อมูล', 'lang' => 'th', 'module' => 'recorddata', 'method' => 'recorddata', 'data_id' => 1, 'status' => 1],
			['type' => 1, 'level' => 1, 'uid' => 'recorddata-recorddata-1', 'slug' => 'recorddata', 'lang' => 'en', 'module' => 'recorddata', 'method' => 'recorddata', 'data_id' => 1, 'status' => 1],

			//personalinformation
			['type' => 1, 'level' => 1, 'uid' => 'personalinformation-personalinformation-1', 'slug' => 'ข้อมูลส่วนบุคคล', 'lang' => 'th', 'module' => 'personalinformation', 'method' => 'personalinformation', 'data_id' => 1, 'status' => 1],
			['type' => 1, 'level' => 1, 'uid' => 'personalinformation-personalinformation-1', 'slug' => 'personalinformation', 'lang' => 'en', 'module' => 'personalinformation', 'method' => 'personalinformation', 'data_id' => 1, 'status' => 1],

			// //catagory
			// ['type' => 1, 'level' => 1, 'uid' => 'products-products_category-1', 'slug' => 'หมวดหมู่สินค้า', 'lang' => 'th', 'module' => 'products', 'method' => 'products_category', 'data_id' => 1, 'status' => 1],
			// ['type' => 1, 'level' => 1, 'uid' => 'products-products_category-1', 'slug' => 'products_category', 'lang' => 'en', 'module' => 'products', 'method' => 'products_category', 'data_id' => 1, 'status' => 1],

			// //contact
			// ['type' => 1, 'level' => 1, 'uid' => 'contact-contact-1', 'slug' => 'ติดต่อเรา', 'lang' => 'th', 'module' => 'contact', 'method' => 'contact', 'data_id' => 1, 'status' => 1],
			// ['type' => 1, 'level' => 1, 'uid' => 'contact-contact-1', 'slug' => 'contact', 'lang' => 'en', 'module' => 'contact', 'method' => 'contact', 'data_id' => 1, 'status' => 1],

			// //content
			// ['type' => 1, 'level' => 1, 'uid' => 'content-content-1', 'slug' => 'บทความ', 'lang' => 'th', 'module' => 'content', 'method' => 'content', 'data_id' => 1, 'status' => 1],
			// ['type' => 1, 'level' => 1, 'uid' => 'content-content-1', 'slug' => 'content', 'lang' => 'en', 'module' => 'content', 'method' => 'content', 'data_id' => 1, 'status' => 1],

			//policy
			['type' => 1, 'level' => 1, 'uid' => 'policy-policy-1', 'slug' => 'นโยบายความเป็นส่วนตัว', 'lang' => 'th', 'module' => 'policy', 'method' => 'policy', 'data_id' => 1, 'status' => 1],
			['type' => 1, 'level' => 1, 'uid' => 'policy-policy-1', 'slug' => 'policy', 'lang' => 'en', 'module' => 'policy', 'method' => 'policy', 'data_id' => 1, 'status' => 1],

		];

		foreach ($slug as $value) {
			Slugs::create($value);
		}
	}
}
