<?php

namespace Modules\Mwz\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Modules\Mwz\app\Models\Geographies;
use Modules\Mwz\app\Models\GeographyTranslate;

class GeographyAdminController extends AdminController
{
    /**
     * Function : construct
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function __construct()
    {
        $this->middleware('auth:admin');

        self::$navbar = [
            ['name' => __("mwz::geo.title"), 'url' => null],
            ['name' => __("mwz::geo.name"), 'url' => route_has("admin.mwz.geo.index")]
        ];
    }

    /**
     * Function : index
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function index()
    {
        return view('mwz::geo.index', ['navbar' => self::$navbar]);
    }

    /**
     * Function : datatable_ajax
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['id', 'name',  'updated_at', 'action'];
            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');

            // create Geographies object
            $req = new Geographies();

            // dt_search 
            if (!empty($dt_search)) {
                $req = $req->where(function ($query) use ($dt_search) {
                    $query->whereRelation('local', 'name', 'like', "%$dt_search%");
                });
            }

            // count all Geographies
            $dt_total = $req->count();
            // set query order & limit from datatable
            $resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
                ->offset($dt_start)
                ->limit($dt_length)
                ->get();

            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('geography_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->editColumn('name', function ($record) {
                    return $record->local->name ?? '';
                })->addColumn('action', function ($record) {
                    $btn = '<div class="button-list">';
                    $btn .= self::btn_route("admin.mwz.geo.edit", $record->id);
                    $btn .= self::btn_delete("admin.mwz.geo.set_delete", $record->id, 'setDelete');
                    $btn .= '</div>';
                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : form
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function form($id = 0)
    {
        $data = [];
        self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
        if (!empty($id)) {
            $data = Geographies::find($id);
            if (!empty($data->langs)) {
                $data->langs = $data->langs->groupBy('lang')->map(function ($row) {
                    return  $row->first();
                });
            }
        }

        return view('mwz::geo.form', ['data' => $data, 'navbar' => self::$navbar]);
    }

    /**
     * Function : save
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function save(Request $request)
    {
        $rule = $msg = [];
        foreach (languages() as $lang => $l) {
            $rule["name_$lang"] = 'required|max:255';
            $msg["name_$lang.*"] = __('mwz::city.field.name_placeholder') . ' ' . ($l['name'] ?? '');
        }
        //validate post data
        $valid = validator($request->all(), $rule, $msg);
        if ($valid->fails()) {
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 404);
        }
        $attr['updated_at'] = now();
        $set = Geographies::updateOrcreate(['id' => $request->get('id')], $attr);
        if ($set->save()) {
            self::setLangs($request, $set->id);
            return self::response(['msg' => __('noti.success')]);
        }

        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : setLangs
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    private function setLangs(Request $request, $id)
    {
        foreach (languages() as $lang => $v) {
            GeographyTranslate::updateOrCreate(
                ['geo_id' => $id, 'lang' => $lang],
                [
                    "name" => mwz_setTextString($request->get("name_{$lang}") ?? ''),
                    'desc' => mwz_setTextString($request->get("desc_{$lang}") ?? ''),
                ]
            );
        }
    }

    /**
     * Function : set_delete
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function set_delete(Request $request)
    {
        $set = Geographies::find($request->get('id'));
        if ($set->delete()) {
            GeographyTranslate::where('geo_id', $request->get('id'))->delete();
            return self::response(['msg' => __('noti.delete')]);
        }
        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : get_list
     * Desc: สำหรับการเรียกข้อมูลใน select2
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function get_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = new Geographies();
        if (!empty($filter['search'])) {
            $req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
        }
        if (!empty($filter['id'])) {
            $req = $req->where('id', '<>', $filter['id']);
        }
        $data = $req->orderBy('id', 'asc')->get();
        $result = [];
        foreach ($data as $list) {
            $result[] = ['id' => $list->id, 'text' => $list->local->name, 'image' =>  ''];
        }
        return self::response(['results' => $result]);
    }


    /**
     * Function : json
     * Desc: สำหรับการเรียกข้อมูลจาก json
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function json()
    {
        $langs = array_keys(languages());
        Geographies::TRUNCATE();
        GeographyTranslate::TRUNCATE();
        $mock = File::json('assets/json/geographies.json');
        foreach ($mock as $item) {
            if (Geographies::where('id', $item['id'])->count() == 0) {
                $req = Geographies::create(['id' => $item['id']]);
                foreach ($langs as $lang) {
                    $req->langs()->updateOrCreate(
                        ['lang' => $lang],
                        ['name' => $item["name"] ?? '']
                    );
                }
            }
        }
        return self::response(['msg' => __('noti.success')]);
    }
}
