<?php

namespace Modules\Mwz\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Modules\Mwz\app\Models\Districts;
use Modules\Mwz\app\Models\DistrictTranslate;
use Modules\Mwz\app\Models\Provinces;

class DistrictAdminController extends AdminController
{
    /**
     * Function : construct
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function __construct()
    {
        $this->middleware('auth:admin');

        self::$navbar = [
            ['name' => __("mwz::district.title"), 'url' => null],
            ['name' => __("mwz::district.name"), 'url' => route_has("admin.mwz.district.index")]
        ];
    }

    /**
     * Function : index
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function index()
    {
        return view('mwz::district.index', ['navbar' => self::$navbar]);
    }

    /**
     * Function : datatable_ajax
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['id', 'name', 'province_id',  'updated_at', 'action'];
            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');

            // create Districts object
            $req = new Districts();
            if (!empty($dt_filter['province_id'])) {
                $req = $req->where('province_id', $dt_filter['province_id']);
            }
            if (!empty($filter['district_id'])) {
                $req = $req->where('district_id', $filter['district_id']);
            }
            // dt_search 
            if (!empty($dt_search))
                $req = $req->where(function ($query) use ($dt_search) {
                    $query->whereRelation('local', 'name', 'like', "%$dt_search%");
                });

            // count all Districts
            $dt_total = $req->count();
            // set query order & limit from datatable
            $resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
                ->offset($dt_start)
                ->limit($dt_length)
                ->get();
            // query Districts

            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('district_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->editColumn('name', function ($record) {
                    return $record->local->name ?? '';
                })->editColumn('province_id', function ($record) {
                    return $record->province->local->name ?? '';
                })->addColumn('action', function ($record) {
                    $btn = '<div class="button-list">';
                    $btn .= self::btn_route("admin.mwz.district.edit", $record->id);
                    $btn .= self::btn_delete("admin.mwz.district.set_delete", $record->id, 'setDelete');
                    $btn .= '</div>';
                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : form
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function form($id = 0)
    {
        $data = [];
        self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
        if (!empty($id)) {
            $data = Districts::find($id);
            if (!empty($data->langs)) {
                $data->langs = $data->langs->groupBy('lang')->map(function ($row) {
                    return  $row->first();
                });
            }
        }

        return view('mwz::district.form', ['data' => $data, 'navbar' => self::$navbar]);
    }

    /**
     * Function : save
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function save(Request $request)
    {
        foreach (languages() as $lang => $l) {
            $rule["name_$lang"] = 'required|max:255';
            $msg["name_$lang.*"] = __('mwz::city.field.name_placeholder') . ' ' . ($l['name'] ?? '');
        }
        $rule["province_id"] = 'required';
        $msg["province_id.*"] = __('mwz::district.field.district_placeholder');
        $valid = validator($request->all(), $rule, $msg);
        if ($valid->fails()) {
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 404);
        }

        $attr['province_id'] = $request->get("province_id");
        $set = Districts::updateOrcreate(['id' => $request->get('id')], $attr);
        if ($set->save()) {
            self::setLangs($request, $set->id);
            return self::response(['msg' => __('noti.success')]);
        }

        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : set translate
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    private function setLangs(Request $request, $id)
    {
        foreach (languages() as $lang => $v) {
            DistrictTranslate::updateOrCreate(
                ['district_id' => $id, 'lang' => $lang],
                ["name" => mwz_setTextString($request->get("name_{$lang}") ?? '')]
            );
        }
    }

    /**
     * Function : set_delete
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function set_delete(Request $request)
    {
        $set = Districts::find($request->get('id'));
        if ($set->delete()) {
            DistrictTranslate::where('district_id', $request->get('id'))->delete();
            return self::response(['msg' => __('noti.delete')]);
        }

        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : get_list
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function get_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = new Districts();
        if (!empty($filter['search'])) {
            $req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
        }
        if (!empty($filter['id'])) {
            $req = $req->where('id', '<>', $filter['id']);
        }
        if (!empty($filter['province_id'])) {
            $req = $req->where('province_id', $filter['province_id']);
        }
        if (!empty($filter['district_id'])) {
            $req = $req->where('district_id', $filter['district_id']);
        }
        $data = $req->orderBy('id', 'asc')->limit(50)->get();
        $result = [];
        foreach ($data as $list) {
            $result[] = ['id' => $list->id, 'text' => $list->local->name ?? null, 'image' =>  ''];
        }

        return self::response(['results' => $result]);
    }

    /**
     * Function : call_list
     * Desc: ดึงข้อมูลอำเภอ
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function call_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = new Districts();
        if (!empty($filter['search'])) {
            $req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
        }
        if (!empty($filter['id'])) {
            $req = $req->where('id', '<>', $filter['id']);
        }
        if (!empty($filter['province_id'])) {
            $req = $req->where('province_id', $filter['province_id']);
        }
        $data = $req->orderBy('id', 'asc')->limit(50)->get();
        $result = [];
        foreach ($data as $list) {
            $result[] = [
                'id' => $list->id, 'text' => $list->local->name ?? null,
                'geo' => ['id' => $list->geo_id, 'text' => $list->geo->local->name ?? null],
                'province' => ['id' => $list->province_id, 'text' => $list->province->local->name ?? null]
            ];
        }

        return self::response(['results' => $result]);
    }

    /**
     * Function : json
     * Desc: ดึงข้อมูลอำเภอจากไฟล์ json
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function json(Request $request)
    {
        $langs = array_keys(languages());
        $index = intval($request->get('index') ?? 1);
        $per_page = 100;
        $offset = ($index - 1) * $per_page;
        $list = array_slice(File::json('assets/json/district.json', JSON_UNESCAPED_UNICODE), $offset, $per_page);
        if (!empty($list)) {
            foreach ($list as $item) {
                if (Districts::where('id', $item['id'])->count() == 0) {
                    $req = Districts::create(['id' => $item['id'], 'province_id' => $item['province_id']]);
                    foreach ($langs as $lang) {
                        $req->langs()->updateOrCreate(
                            ['lang' => $lang],
                            ['name' => $item["name_{$lang}"] ?? $item['name_th'] ?? '']
                        );
                    }
                    $req->geo_id = $req->province->geo_id;
                    $req->save();
                }
            }
            return self::response(['status' => 0]);
        }
        return self::response(['status' => 1]);
    }
}
