<?php

namespace Modules\Menu\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Modules\Menu\app\Models\Menu;
use Modules\Menu\app\Models\MenuTranslate;
use Modules\Mwz\app\Http\Controllers\AdminController;
use SebastianBergmann\CodeCoverage\Report\Xml\Report;

class MenuAdminController extends AdminController
{
	private $config, $action;

	/**
	 * Function : construct
	 * Dev : Peh
	 * Update Date : 29 Apr 2024
	 */
	public function __construct(Request $request)
	{
		$routeName = Route::currentRouteName();
		if ($routeName) {
			$expo = explode('.', $routeName);
			$this->action = $expo[3] ?? '';
		} else {
			$this->action = '';
		}

		self::$navbar[] =  ['name' => __("menu::menu.title"), 'url' => route_has("admin.menu.group.index")];

		$this->config = config("menu.menu");
	}

	/**
	 * Function : menu index
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $group_id
	 * @return \Illuminate\Contracts\View\View menu::menu.index
	 */
	public function index(Request $request, $group_id = 0)
	{
		self::$navbar[] =  ['name' => __("menu::menu.name"), 'url' => route('admin.menu.menu.index', $group_id)];
		$root = Menu::whereNull('parent_id')->get();

		return view("menu::menu.index", ['config' => $this->config, 'navbar' => self::$navbar, 'root' => $root, 'group_id' => $group_id]);
	}

	/**
	 * Function : menu datatable_ajax
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $group_id
	 * @return Yajra\DataTables\Facades\DataTables data table
	 */
	public function datatable_ajax(Request $request, $group_id = 0)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['_lft', 'name'];
			if ($this->config['setting']['parent']['status'])
				array_push($dt_column, 'parent_id');
			if ($this->config['setting']['widget']['status'])
				array_push($dt_column, 'widget');
			if ($this->config['setting']['style']['status'])
				array_push($dt_column, 'style');
			array_push($dt_column, 'sort', 'updated_at', 'action');

			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create menu object
			$req = new Menu();

			$req = $req->whereDescendantOf($group_id);

			// dt_search 
			if (!empty($dt_search))
				$req = $req->whereRelation('local', 'name', 'like', "%{$dt_search}%");

			// count all menu 
			$dt_total = $req->count();
			// set query order & limit from datatable
			$data = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)
				->limit($dt_length)
				->withDepth()
				->defaultOrder()
				->get();

			// prepare datatable for response
			$tables = datatables($data)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('menu_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('name', function ($record) {
					return str_repeat(' - ', $record->depth - 1) . ($record->local->name ?? '');
				})->editColumn('parent_id', function ($record) {
					return $record->parent->local->name ?? '-';
				})
				// ->editColumn('link', function ($record) {
				// 	$link = '';
				// 	switch ($record->type) {
				// 		case 1:
				// 			$link = 'Slug: ' . ($record->slug->slug ?? '');
				// 			break;
				// 		case 2:
				// 			$link = 'URL: ' . $record->url;
				// 			break;
				// 		case 3:
				// 			$link = 'widget: ' . ($this->config['setting']['widgets'][$record->widget] ?? '');
				// 			break;
				// 	}
				// 	return $link . (!empty($record->tag) ? '#' . $record->tag : '');
				// })
				->editColumn('widget', function ($record) {
					return $this->config['setting']['widgets'][$record->widget] ?? '-';
				})->editColumn('style', function ($record) {
					return $record->style ?? '-';
				})->editColumn('sort', function ($record) use ($dt_order) {
					return self::btn_sort("admin.menu.menu.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->editColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					if (!empty($this->config['btn']['status']))
						$btn .= self::btn_status("admin.menu.menu.set_status", $record->id, 'setStatus', $record->status);
					if (!empty($this->config['btn']['edit']))
						$btn .= self::btn_route("admin.menu.menu.edit", [$record->parent_id, $record->id]);
					if (!empty($this->config['btn']['delete']))
						$btn .= self::btn_delete("admin.menu.menu.set_delete", $record->id, 'setDelete');
					$btn .= '</div>';

					return $btn;
				})->escapeColumns([]);

			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : menu form
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param int $group_id
	 * @param int $id
	 * @return \Illuminate\Contracts\View\View menu::menu.form
	 */
	public function form($group_id = 0, $id = 0)
	{
		self::$navbar[] =  ['name' => __("menu::menu.name"), 'url' => route('admin.menu.menu.index', $group_id)];
		self::$navbar[] =  ['name' => __("action.{$this->action}"), 'url' => null];
		$data = null;
		if (!empty($id)) {
			$data = Menu::where([['id', $id], ['parent_id', $group_id]])->first();
			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}

			if (empty($data))
				return abort(Response::HTTP_NOT_FOUND);
		}

		return view('menu::menu.form', [
			'data' => $data,
			'root_id' => !empty($data) ? self::check_root($data) : $group_id,
			'group_id' => $group_id,
			'navbar' => self::$navbar,
			'config' => $this->config,
			'action' => $this->action
		]);
	}

	/**
	 * Function : check root
	 * Dev : Petch
	 * Update Date : 10 Feb 2024
	 */
	private static function check_root($data)
	{
		if (!$data->isRoot())
			return self::check_root($data->parent);

		return $data->id;
	}

	/**
	 * Function : menu save
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function save(Request $request)
	{
		//validate post data
		$rule = $msg = [];
		foreach (languages() as $lang => $l) {
			foreach ($this->config['input'] as $key => $item) {
				if (!empty($item['status']) && !empty($item['validate'])) {
					$rule["{$key}_{$lang}"] = ($key != 'name') ? 'required' : 'required|max:255';
					$msg["{$key}_{$lang}.*"] = __("field.{$key}_placeholder") . ' ' . ($l['name'] ?? '');
				}
			}
		}
		foreach ($this->config['setting'] as $key => $item) {
			if (!empty($item['status']) && !empty($item['validate'])) {
				$rule["{$key}_{$lang}"] =  'required';
				$msg["{$key}_{$lang}.*"] = __("field.{$key}_placeholder");
			}
		}
		//validate post data
		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_MOVED_PERMANENTLY);

		$attr = [
			"parent_id" => $request->get('group_id') ?? 0,
			"status" => $request->get('status') ?? 0
		];
		if (!empty($this->config['setting']['type']['status'])) {
			$attr['type'] = $request->get("type");
		}
		if (!empty($this->config['setting']['url']['status'])) {
			$attr['url'] = $request->get("url");
		}
		if (!empty($this->config['setting']['slug']['status'])) {
			$attr['slug_id'] = $request->get("slug_id");
		}
		if (!empty($this->config['setting']['widget']['status'])) {
			$attr['widget'] = $request->get("widget");
		}
		if (!empty($this->config['setting']['parent']['status']) && !empty($request->get("parent_id"))) {
			$attr['parent_id'] = $request->get("parent_id");
		}
		if (!empty($this->config['setting']['tag']['status'])) {
			$attr['tag'] = $request->get("tag");
		}
		if (!empty($this->config['setting']['style']['status'])) {
			$attr['style'] = $request->get("style");
		}

		$set = Menu::updateOrCreate(['id' => $request->get('id') ?? 0], $attr);
		if ($set->save()) {
			self::re_order();
			self::setLangs($request, $set);
			return self::response(['msg' => __('noti.success')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : menu set translate
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $id
	 * @return void
	 */
	private function setLangs(Request $request, $row)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($this->config['input'] as $key => $item) {
				$attr[$key] = mwz_setTextString($request->get("{$key}_{$lang}") ?? '', $item['type'] == 'texteditor');
			}
			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}
	/**
	 * Function : menu set status
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_status(Request $request)
	{
		$set = Menu::find($request->get('id'));
		$set->status = !$set->status;
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : menu set delete
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_delete(Request $request)
	{
		$set = Menu::find($request->get('id'));
		if ($set->delete()) {
			MenuTranslate::where('menu_id', $request->get('id'))->delete();
			self::re_order();
			return self::response(['msg' => __('noti.delete')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : menu re order
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @return void
	 */
	public function re_order()
	{
		Menu::fixTree();
	}

	/**
	 * Function : menu tree
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $group_id
	 * @return \Illuminate\Contracts\View\View menu::menu.tree
	 */
	public function tree(Request $request, $group_id = 0)
	{
		self::$navbar[] =  ['name' => __('action.sort'), 'url' => null];
		$data = Menu::whereDescendantOf($group_id)->orderBy('_lft')->get()->toTree();

		return view('menu::menu.tree', ['navbar' => self::$navbar, 'data' => $data, 'group_id' => $group_id, 'config' => $this->config]);
	}

	/**
	 * Function : menu tree sort
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $group_id
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function tree_sort(Request $request, $group_id = 0)
	{
		try {
			//code...
			if (empty($request->id)) {
				$resp = ['msg' => 'no id'];
				$status = Response::HTTP_MULTIPLE_CHOICES;
			} else {
				$menu = Menu::find($request->id);
				if (empty($menu)) {
					$resp = ['msg' => 'no data'];
					$status = Response::HTTP_MULTIPLE_CHOICES;
				} else {
					$parent = Menu::find($request->parent_id);
					if (empty($parent)) {
						$menu->parent_id = $group_id;
						$neighbor = Menu::where('parent_id', $group_id)->orderBy('_lft', 'asc')->get();
						if ($neighbor->count() > $request->position && isset($neighbor[$request->position])) {
							if ($request->position > $request->old_position) {
								$menu->afterNode($neighbor[$request->position])->save();
							} else {
								$menu->beforeNode($neighbor[$request->position])->save();
							}
						} else {
							// Position out of bounds, just save with group_id as parent
							$menu->parent_id = $group_id;
							$menu->save();
						}
						$resp = ['msg' => __('noti.success'), 'data' => $neighbor];
						$status = Response::HTTP_OK;
					} else {
						$main_id = $parent->id;
						$old_parent_id = $menu->parent_id;
						$neighbor = Menu::where('parent_id', $main_id)->orderBy('_lft', 'asc')->get();
						if (empty($neighbor)) {
							$menu->parent_id = $main_id;
							$menu->save();
							$resp = ['msg' => __('noti.success')];
							$status = Response::HTTP_OK;
						} else {
							if ($old_parent_id == $main_id) {
								if ($neighbor->count() > $request->position && isset($neighbor[$request->position])) {
									if ($request->position > $request->old_position) {
										$menu->afterNode($neighbor[$request->position])->save();
									} else {
										$menu->beforeNode($neighbor[$request->position])->save();
									}
								} else {
									// Position out of bounds, just save with parent
									$menu->parent_id = $main_id;
									$menu->save();
								}
							} else {
								if ($neighbor->count() > $request->position && isset($neighbor[$request->position])) {
									$menu->beforeNode($neighbor[$request->position])->save();
								} else {
									// Position out of bounds, just save with parent
									$menu->parent_id = $main_id;
									$menu->save();
								}
							}
							$resp = ['msg' => __('noti.success'), 'data' => $neighbor];
							$status = Response::HTTP_OK;
						}
					}
				}
			}
			self::re_order();
		} catch (\Throwable $th) {
			$resp = ['msg' => __('noti.error')];
			$status = Response::HTTP_BAD_REQUEST;
		}

		return self::response($resp, $status);
	}

	/**
	 * Function : menu set sort
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $group_id
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_sort(Request $request, $group_id = 0)
	{
		$node = Menu::find($request->get('id'));
		$move = false;
		if ($request->get('move') == 'up') {
			$move = $node->up();
		} else {
			$move = $node->down();
		}
		if ($move) {
			self::re_order();

			return self::response(['msg' => __('noti.success')]);
		}

		return self::response(['msg' => __('noti.sort_error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : menu get list
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = new Menu();
		$req = $req->whereNull('parent_id');
		if (!empty($filter['search'])) {
			$req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
		}
		if (!empty($filter['id'])) {
			$req = $req->where('id', '<>', $filter['id']);
		}
		if (!empty($filter['group_id'])) {
			$req = $req->withDepth()->defaultOrder()->descendantsAndSelf($filter['group_id']);
		} else {
			$req = $req->withDepth()->defaultOrder()->orderBy('_lft')->get();
		}
		$result = [];
		foreach ($req as $list) {
			$result[] = ['id' => $list->id, 'text' => str_repeat(' - ', $list->depth) . $list->local->name, 'image' =>  ''];
		}

		return self::response(['msg' => __('noti.success'), 'results' => $result]);
	}
}
