<?php

namespace Modules\Menu\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Modules\Menu\app\Models\Menu;
use Modules\Menu\app\Models\MenuTranslate;
use Modules\Mwz\app\Http\Controllers\AdminController;

class GroupMenuAdminController extends AdminController
{
	private $config;

	/**
	 * Function : construct
	 * Dev : Peh
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{

		self::$navbar[] =  ['name' => __("menu::menu.title"), 'url' => route_has("admin.menu.group.index")];

		$this->config = config("menu.group");
	}

	/**
	 * Function : group menu index
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @return \Illuminate\Contracts\View\View menu::group.index
	 */
	public function index()
	{
		return view("menu::group.index", ['config' => $this->config, 'navbar' => self::$navbar]);
	}

	/**
	 * Function : menu datatable ajax
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param Request $request
	 * @return Yajra\DataTables\Facades\DataTables data table
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = array('_lft', 'name', 'updated_at', 'action');
			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create menu object
			$req = new Menu();

			// get root only
			$req = $req->whereNull('parent_id');

			// dt_search 
			if (!empty($dt_search))
				$req = $req->whereRelation('local', 'name', 'like', "%{$dt_search}%");

			// count all menu
			$dt_total = $req->count();

			// set query order & limit from datatable
			$data = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)->limit($dt_length)
				->withDepth()->defaultOrder()->get();

			// prepare datatable for response
			$tables = datatables($data)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('menu_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('name', function ($record) {
					return str_repeat(' - ', $record->depth) . ($record->local->name ?? '');
				})->editColumn('sort', function ($record) {
					return self::btn_sort("admin.menu.group.set_sort", $record->id, 'setSort');
				})->editColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					if (!empty($this->config['btn']['status'])) {
						$btn .= self::btn_status("admin.menu.group.set_status", $record->id, 'setStatus', $record->status);
					}
					$btn .= $this->route_button(mwz_route("admin.menu.menu.index", [$record->id]), $icon = 'fe-menu', $target = '_self');
					if (!empty($this->config['btn']['edit'])) {
						$btn .= self::btn_route("admin.menu.group.edit", $record->id);
					}
					if (!empty($this->config['btn']['delete'])) {
						$btn .= self::btn_delete("admin.menu.group.set_delete", $record->id, 'setDelete');
					}
					$btn .= '</div>';
					return $btn;
				})
				->escapeColumns([]);
			// response datatable json

			return $tables->make(true);
		}
	}

	/**
	 * Function : menu form
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param int $id
	 * @return \Illuminate\Contracts\View\View menu::group.form
	 */
	public function form($id = 0)
	{
		self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
		$data = null;
		if (!empty($id)) {
			$data = Menu::find($id);
			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}
		}

		return view("menu::group.form", ['data' => $data, 'navbar' => self::$navbar,  'config' => $this->config]);
	}

	/**
	 * Function : menu save
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function save(Request $request)
	{
		$rule = $msg = [];
		foreach (languages() as $lang => $l) {
			foreach ($this->config['input'] as $key => $item) {
				if (!empty($item['status']) && !empty($item['validate'])) {
					$rule["{$key}_{$lang}"] = ($key != 'name') ? 'required' : 'required|max:255';
					$msg["{$key}_{$lang}.*"] = __("field.{$key}_placeholder") . ($l['name'] ?? "");
				}
			}
		}
		//validate post data
		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_MOVED_PERMANENTLY);

		$menu_count = Menu::whereNull('parent_id')->count();
		if (empty($request->get('id')) && $menu_count >= $this->config['max'])
			return self::response(['msg' => __('noti.error')], Response::HTTP_MULTIPLE_CHOICES);

		$attr = [
			"status" => $request->get('status') ?? 0
		];
		$menu = Menu::updateOrCreate(['id' => $request->get('id') ?? 0], $attr);
		if ($menu->save()) {
			self::setLangs($request, $menu);

			return self::response(['msg' => __('noti.success')], Response::HTTP_OK);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : menu translate
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $id
	 * @return void
	 */
	private function setLangs(Request $request, $row)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($this->config['input'] as $key => $item) {
				$attr[$key] = mwz_setTextString($request->get("{$key}_{$lang}") ?? '', $item['type'] == 'texteditor');
			}
			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}

	/**
	 * Function : menu set status
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_status(Request $request)
	{
		$set = Menu::find($request->get('id'));
		$set->status = !$set->status;
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')], Response::HTTP_OK);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : menu set delete
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_delete(Request $request)
	{
		$set = Menu::find($request->get('id'));
		if ($set->delete()) {
			self::re_order();
			return self::response(['msg' => __('noti.delete')], Response::HTTP_OK);
		}
		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	public function re_order()
	{
		Menu::fixTree();
	}

	/**
	 * Function : menu set sort
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_sort(Request $request)
	{
		$node = Menu::find($request->get('id'));
		$move = false;
		if ($request->get('move') == 'up')
			$move = $node->up();
		else
			$move = $node->down();
		if ($move) {
			self::re_order();
			return self::response(['msg' => __('noti.success')], Response::HTTP_OK);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : menu get list
	 * Dev : Peh
	 * Update Date : 18 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = new Menu();
		if (!empty($filter['search'])) {
			$req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
		}
		if (!empty($filter['id'])) {
			$req = $req->where('id', '<>', $filter['id']);
		}

		$req = $req->withDepth()->defaultOrder()->orderBy('_lft')->get();
		$result = [];
		foreach ($req as $list) {
			$result[] = ['id' => $list->id, 'text' => str_repeat(' - ', $list->depth) . $list->local->name, 'image' =>  ''];
		}
		return self::response(['msg' => __('noti.success'), 'results' => $result], Response::HTTP_OK);
	}
}
