<?php

namespace Modules\Member\app\Http\Controllers;

use App\Exports\MemberExport;
use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Carbon\Carbon;
use Modules\Member\app\Models\MemberTimeline;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Frontend\app\Http\Controllers\FrontendInitController;
use Modules\Map\app\Models\CheckInOut;
use Modules\Member\app\Models\Member;

class TimelineAdminController extends AdminController
{
    private $module, $method, $action, $config;

    /**
     * Function : construct
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    public function __construct()
    {
        $routeName = Route::currentRouteName();
		if ($routeName) {
			$explode = explode('.', $routeName);
			$this->module = $explode[1] ?? '';
			$this->method = $explode[2] ?? '';
			$this->action = $explode[3] ?? '';
		} else {
			$this->module = '';
			$this->method = '';
			$this->action = '';
		}

        self::$navbar = [
            ['name' => __("member::timeline.{$this->module}.title"), 'url' => null],
            ['name' => __("member::timeline.{$this->module}.name"), 'url' => route_has("admin.{$this->module}.{$this->method}.index")]
        ];
        $this->config = config("member.timeline.{$this->module}");
    }


    /**
     * Display a listing of the resource.
     */
    public function index($member_id = 0)
    {
         $limit = false;
        if (!empty($this->config['setting']['limit']['status']))
            $limit = CheckInOut::where('status', 1)->count() >= $this->config['setting']['limit']['max'];
        if (!empty($member_id)) {
            $member = Member::find($member_id);
        }
        
        return view('member::timeline.index',[
            'navbar' => self::$navbar,
            'config' => $this->config,
            'method' => $this->method,
            'type' => $this->module,
            'limit' => $limit,
            'selectedMember' => $member ?? null,
        ]);
    }

    public function form(Request $request,$id = 0)
    {
        $data = null;

        if (!empty($id)) {
            $data = CheckInOut::find($id);
        }
        return view('member::timeline.form', [
            'type' => $this->module,
            'method' => $this->method,
            'config' => $this->config,
            'navbar' => self::$navbar,
            'action' => $this->action,
            'data' => $data,
        ]);
    }


    /**
     * Function : construct
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['id'];
            foreach ($this->config['table']['header'] as $h) {
                if (!empty($h['status']))
                    $dt_column[] = $h['column'];
            }
            if (!empty($this->config['table']['btn']['sort']))
                $dt_column[] = 'sort';
            array_push($dt_column, 'updated_at', 'action');

            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');
            // create Category object
            $req = new CheckInOut;
            // dt_search 
            if (!empty($dt_search)) {
                $req = $req->where(function($query) use ($dt_search) {
                    $query->orWhere('current_place', 'like', "%{$dt_search}%")
                          ->orWhere('target_place', 'like', "%{$dt_search}%")
                          ->orWhereRelation('member', 'name', 'like', "%{$dt_search}%");
                });
            }
            
            // parent_id
			if (!empty($dt_filter['period'])) {
				$range = !empty($dt_filter['range']) ? $dt_filter['range'] : null;
				$req = self::mwz_filter($req, $dt_filter['period'], 'created_at', $range);
			}

            // dt_filter - date period
            if (!empty($dt_filter['check_in_out_period'])) {
                if (strpos($dt_filter['check_in_out_period'], ' to ') === false) {
                    // กรณีกรอกแค่วันเดียว เช่น "2025-10-27"
                    $singleDate = $dt_filter['check_in_out_period'];
                    $req = $req->whereDate('created_at', $singleDate);
                } else {
                    // ช่วงวันที่
                    $req = self::mwz_filter($req, $dt_filter['check_in_out_period'], 'created_at', $dt_filter['range'] ?? null);
                }
            }
            
            // dt_filter - check in/out period (วันที่และเวลาเข้า-ออก)
            if (!empty($dt_filter['check_in_out_period'])) {
                $dates = explode(' to ', $dt_filter['check_in_out_period']);
                if (count($dates) == 2) {
                    $req = $req->whereBetween('time', [
                        $dates[0] . ' 00:00:00',
                        $dates[1] . ' 23:59:59'
                    ]);
                }
            }
            
            // dt_filter - member
            if (!empty($dt_filter['member_id']))
                $req = $req->where('member_id', $dt_filter['member_id']);
            
            // dt_filter - distance_km (ระยะทาง)
            if (!empty($dt_filter['distance_km_min']) && is_numeric($dt_filter['distance_km_min']))
                $req = $req->where('distance_km', '>=', $dt_filter['distance_km_min']);
            
            if (!empty($dt_filter['distance_km_max']) && is_numeric($dt_filter['distance_km_max']))
                $req = $req->where('distance_km', '<=', $dt_filter['distance_km_max']);
            
            // dt_filter - oil_price (ค่าน้ำมัน)
            if (!empty($dt_filter['oil_price_min']) && is_numeric($dt_filter['oil_price_min']))
                $req = $req->where('oil_price', '>=', $dt_filter['oil_price_min']);
            
            if (!empty($dt_filter['oil_price_max']) && is_numeric($dt_filter['oil_price_max']))
                $req = $req->where('oil_price', '<=', $dt_filter['oil_price_max']);

            // count all 
            $dt_total = $req->count();

            // set query order & limit from datatable
            $resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
                ->offset($dt_start)
                ->limit($dt_length)
                ->get();
            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('check_in_out_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('member_id', function ($record) {
                    return $record->member->name ?? '';
                })->editColumn('current_place', function ($record) {
                    return $record->current_place ?? '';
                })->editColumn('target_place', function ($record) {
                    return $record->target_place ?? '';
                })->editColumn('distance_km', function ($record) {
                    return $record->distance_km ?? '';
                })->editColumn('oil_price', function ($record) {
                    return $record->oil_price ?? '';
                })->editColumn('created_at', function ($record) {
                    return $record->created_at ? $record->created_at->format('d/m/Y') : '';
                })->editColumn('time', function ($record) {
                    return $record->time ? $record->time->format('H:i') : '';
                })->editColumn('checkout_time', function ($record) {
                    return $record->checkout_time ? $record->checkout_time->format('H:i') : '';
                })->editColumn('status', function ($record) use ($dt_order) {
                    return $record->status ? $this->status_appove($record->status) : '-';
                })->editColumn('sort', function ($record) use ($dt_order) {
                    if (!empty($this->config['table']['btn']['sort']))
                        return self::btn_sort("admin.{$this->module}.{$this->method}.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
                })->editColumn('updated_at', function ($record) {
                    return $record->updated_at ? str_replace(' ', '<br>', $record->updated_at->format('Y-m-d H:i:s')) : '';
                })->addColumn('action', function ($record) {
                    $btn = '<div class="button-list">';
                    if (!empty($this->config['table']['btn']['default']))
                        $btn .= self::btn_default("admin.{$this->module}.{$this->method}.set_default", $record->id, 'setDefault', $record->default);
                    if (!empty($this->config['table']['btn']['status']))
                        $btn .= self::btn_status("admin.{$this->module}.{$this->method}.set_status", $record->id, 'setStatus', $record->status);
                    if (!empty($this->config['table']['btn']['edit']))
                        $btn .= self::btn_route("admin.{$this->module}.{$this->method}.edit", $record->id,'btn-white','fe-eye');
                    if (!empty($this->config['table']['btn']['delete']))
                        $btn .= self::btn_delete("admin.{$this->module}.{$this->method}.set_delete", $record->id, 'setDelete');

                    $btn .= '</div>';

                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    
	private function status_appove($status)
	{
		if (!empty($status)) {
			if ($status == 1) return '<span class="badge bg-warning text-dark">รอส่งคำขอ</span>';
			if ($status == 2) return '<span class="badge bg-warning text-white align-middle">รออนุมัติ</span>'; // manager
			if ($status == 3) {
				return '<span class="badge bg-warning text-white align-middle">รออนุมัติ</span>'; // BU
			}
			if ($status == 4) return '<span class="badge bg-success text-white align-middle">อนุมัติ</span>'; // BU
			if ($status == 13) return '<span class="badge bg-danger text-white align-middle">ไม่อนุมัติ</span>';
		}
		return '';
	}

    /**
     * Function : export timeline to excel or pdf
     * Dev : AI Assistant
     * Update Date : 07 Oct 2025
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'excel'); // Default to excel
        $filter = $request->get('filter', []);
        $timelines = $this->buildTimelineQuery($request)->get();
        $timestamp = date('Y-m-d_His');
        
        if ($format === 'pdf') {
            $pdf = Pdf::loadView('member::exports.check-in-out-pdf', compact('timelines'))
                ->setPaper('a4', 'landscape');
            $filename = 'timeline_' . $timestamp . '.pdf';
            return $pdf->download($filename);
        } else {
            // Excel export
            $filename = 'timeline_' . $timestamp . '.xlsx';
            return Excel::download(new \App\Exports\TimelineExport($timelines), $filename);
        }
    }

    /**
     * Apply table filters to a base query so exports and tables stay in sync.
     */
    private function buildTimelineQuery(Request $request)
    {
        $query = CheckInOut::with('member');

        $period = $this->getFilterValue($request, 'period');
        if ($period !== null && $period !== '' && $period !== '00') {
            $range = $this->getFilterValue($request, 'range_period');
            $query = self::mwz_filter($query, (int) $period, 'created_at', $range);
        }

        $checkPeriod = $this->getFilterValue($request, 'check_in_out_period');
        if (!empty($checkPeriod)) {
            if (strpos($checkPeriod, ' to ') === false) {
                $query->whereDate('created_at', $checkPeriod);
            } else {
                [$start, $end] = array_map('trim', explode(' to ', $checkPeriod));
                $query->whereBetween('time', [
                    Carbon::parse($start)->startOfDay(),
                    Carbon::parse($end)->endOfDay(),
                ]);
            }
        }

        $memberId = $this->getFilterValue($request, 'member_id');
        if (!empty($memberId)) {
            $query->where('member_id', $memberId);
        }

        $distanceMin = $this->getFilterValue($request, 'distance_km_min');
        if ($distanceMin !== null && $distanceMin !== '' && is_numeric($distanceMin)) {
            $query->where('distance_km', '>=', $distanceMin);
        }

        $distanceMax = $this->getFilterValue($request, 'distance_km_max');
        if ($distanceMax !== null && $distanceMax !== '' && is_numeric($distanceMax)) {
            $query->where('distance_km', '<=', $distanceMax);
        }

        $oilMin = $this->getFilterValue($request, 'oil_price_min');
        if ($oilMin !== null && $oilMin !== '' && is_numeric($oilMin)) {
            $query->where('oil_price', '>=', $oilMin);
        }

        $oilMax = $this->getFilterValue($request, 'oil_price_max');
        if ($oilMax !== null && $oilMax !== '' && is_numeric($oilMax)) {
            $query->where('oil_price', '<=', $oilMax);
        }
        return $query->orderBy('id', 'asc');
        // return $query->orderBy('submission_date', 'desc');
    }

    /**
     * Retrieve a filter value regardless of whether it came through
     * the nested filter[] inputs or as a standalone query parameter.
     */
    private function getFilterValue(Request $request, string $key, $default = null)
    {
        $filters = $request->input('filter', []);
        if (array_key_exists($key, $filters)) {
            $value = $filters[$key];
            if ($value !== '' && $value !== null) {
                return $value;
            }
        }

        $direct = $request->input($key, $default);
        return ($direct === '' ? $default : $direct);
    }

    /**
     * Clear timeline records older than selected months
     */
    public function clear(Request $request)
    {
        $months = (int) $request->get('months', 1);
        if ($months < 1 || $months > 12) {
            return self::response(['msg' => __('noti.error') . ' : invalid months'], Response::HTTP_BAD_REQUEST);
        }

        $thresholdDate = date('Y-m-d', strtotime("-{$months} months"));

        $deleted = CheckInOut::whereDate('submission_date', '<', $thresholdDate)->delete();

        return self::response([
            'msg' => __('noti.success'),
            'deleted' => $deleted,
            'threshold' => $thresholdDate,
        ]);
    }

}
