/**
 * Record Data Manager - Check-in/Check-out System
 */

let currentCheckin = null;

/**
 * Utility Functions
 */
function toDatetimeLocal(date) {
  const pad = n => n.toString().padStart(2, "0");
  return `${date.getFullYear()}-${pad(date.getMonth() + 1)}-${pad(date.getDate())}T${pad(date.getHours())}:${pad(date.getMinutes())}`;
}

function showElement(el) {
  el.classList.remove("hidden");
}

function hideElement(el) {
  el.classList.add("hidden");
}

function clearErrors() {
  document.querySelectorAll(".error-msg").forEach(el => el.textContent = "");
}

function resetToInitialStep() {
  currentCheckin = null;
  hideElement(document.getElementById("checkinForm"));
  hideElement(document.getElementById("checkoutForm"));
  hideElement(document.getElementById("checkoutPreview"));
  hideElement(document.getElementById("checkoutEditForm"));
  hideElement(document.getElementById("latestCheckinBox"));
  hideElement(document.getElementById("btnCheckout"));
  showElement(document.getElementById("btnCheckin"));
  document.querySelectorAll('#checkinForm input,#checkinForm textarea').forEach(el => el.value = "");
  document.querySelectorAll('#checkoutForm input,#checkoutForm textarea').forEach(el => el.value = "");
  document.querySelectorAll('#checkoutEditForm input').forEach(el => el.value = "");
}

/**
 * Haversine formula - Calculate distance between two coordinates
 */
function getDistanceKm(lat1, lon1, lat2, lon2) {
  const R = 6371; // Earth radius in km
  const dLat = (lat2 - lat1) * Math.PI / 180;
  const dLon = (lon2 - lon1) * Math.PI / 180;
  const a = Math.sin(dLat / 2) ** 2 + Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * Math.sin(dLon / 2) ** 2;
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  return R * c;
}

/**
 * Update preview distance
 */
function updatePreviewDistance() {
  const lat = currentCheckin.lat;
  const lng = currentCheckin.lng;
  const targetLat = currentCheckin.lat;
  const targetLng = currentCheckin.lng;
  const distance = getDistanceKm(lat, lng, targetLat, targetLng).toFixed(2);
  document.getElementById("previewDistance").textContent = distance;
  document.getElementById("editDistance").textContent = distance;
}

/**
 * Geolocation Functions
 */
function getCurrentLocation() {
  if (navigator.geolocation) {
    navigator.geolocation.getCurrentPosition(pos => {
      const lat = pos.coords.latitude;
      const lng = pos.coords.longitude;
      document.getElementById("lat").value = lat.toFixed(6);
      document.getElementById("lng").value = lng.toFixed(6);
      
      const geocoder = new google.maps.Geocoder();
      geocoder.geocode({ location: { lat, lng } }, (results, status) => {
        if (status === "OK" && results[0]) {
          document.getElementById("address").value = results[0].formatted_address;
        } else {
          document.getElementById("address").value = "";
          document.getElementById("locationModalMessage").textContent = "เกิดข้อผิดพลาดในการดึงที่อยู่: " + status;
          showElement(document.getElementById("locationModal"));
        }
      });
    }, err => {
      document.getElementById("address").value = "";
      document.getElementById("locationModalMessage").textContent = "ไม่สามารถดึงตำแหน่งได้: " + err.message;
      showElement(document.getElementById("locationModal"));
    });
  } else {
    document.getElementById("address").value = "";
    document.getElementById("locationModalMessage").textContent = "เบราว์เซอร์ของคุณไม่รองรับ Geolocation";
    showElement(document.getElementById("locationModal"));
  }
}

/**
 * Google Maps Autocomplete
 */
function initAutocomplete() {
  new google.maps.places.Autocomplete(document.getElementById("placeName"));
}

// Make initAutocomplete available globally for Google Maps callback
window.initAutocomplete = initAutocomplete;

/**
 * Save Checkout Data
 */
function saveCheckoutData(data) {
  clearErrors();
  const { currentPlace, targetPlace, reason, checkinTime, checkoutTime, lat, lng, targetLat, targetLng } = data;
  let hasError = false;

  if (!currentPlace) {
    document.getElementById("error-currentPlaceOut").textContent = "กรอกชื่อสถานที่ปัจจุบัน";
    hasError = true;
  }
  if (!targetPlace) {
    document.getElementById("error-targetPlaceOut").textContent = "กรอกชื่อสถานที่จะไป";
    hasError = true;
  }
  if (!reason) {
    document.getElementById("error-reasonOut").textContent = "กรอกเหตุผล";
    hasError = true;
  }
  if (!checkoutTime) {
    document.getElementById("error-checkoutTimeOut").textContent = "เลือกวันที่และเวลา";
    hasError = true;
  }
  if (hasError) return;

  let distance = '-';
  if (lat && lng && targetLat && targetLng) {
    distance = getDistanceKm(lat, lng, targetLat, targetLng).toFixed(2);
  }

  const tbody = document.getElementById("history-body");
  const tr = document.createElement("tr");
  tr.innerHTML = `<td>${new Date(checkoutTime).toLocaleDateString()}</td>
                <td>${new Date(checkinTime).toLocaleTimeString()}</td>
                <td>${new Date(checkoutTime).toLocaleTimeString()}</td>
                <td>${currentPlace} → ${targetPlace}</td>
                <td>${distance}</td>`;
  tbody.appendChild(tr);
  showElement(document.getElementById("checkoutModal"));
}

/**
 * Event Listeners - Initialize when DOM is ready
 */
document.addEventListener('DOMContentLoaded', function() {
  
  // Check-in button
  document.getElementById("btnCheckin").addEventListener("click", () => {
    showElement(document.getElementById("checkinForm"));
    getCurrentLocation();
    document.getElementById("checkinTime").value = toDatetimeLocal(new Date());
  });

  // Save Check-in
  document.getElementById("saveCheckin").addEventListener("click", () => {
    clearErrors();
    const currentPlace = document.getElementById("currentPlace").value.trim();
    const targetPlace = document.getElementById("targetPlace").value.trim();
    const reason = document.getElementById("reason").value.trim();
    const placeName = document.getElementById("placeName").value.trim();
    const lat = parseFloat(document.getElementById("lat").value.trim());
    const lng = parseFloat(document.getElementById("lng").value.trim());
    const address = document.getElementById("address").value.trim();
    const time = document.getElementById("checkinTime").value;
    let hasError = false;

    if (!currentPlace) {
      document.getElementById("error-currentPlace").textContent = "ใส่ชื่อสถานที่ปัจจุบัน";
      hasError = true;
    }
    if (!targetPlace) {
      document.getElementById("error-targetPlace").textContent = "ใส่ชื่อสถานที่จะไป";
      hasError = true;
    }
    if (!reason) {
      document.getElementById("error-reason").textContent = "ใส่เหตุผล";
      hasError = true;
    }
    if (!time) {
      document.getElementById("error-checkinTime").textContent = "เลือกวันที่และเวลา";
      hasError = true;
    }
    if (hasError) return;

    currentCheckin = { currentPlace, targetPlace, reason, placeName, lat, lng, address, time };

    document.getElementById("latestLat").textContent = currentCheckin.lat;
    document.getElementById("latestLng").textContent = currentCheckin.lng;
    showElement(document.getElementById("latestCheckinBox"));

    document.getElementById("previewCurrentPlace").textContent = currentCheckin.currentPlace;
    document.getElementById("previewTargetPlace").textContent = currentCheckin.targetPlace;
    document.getElementById("previewReason").textContent = currentCheckin.reason;
    document.getElementById("previewCheckinTime").textContent = new Date(currentCheckin.time).toLocaleString();
    document.getElementById("checkinTimeOutPreview").value = toDatetimeLocal(new Date());

    updatePreviewDistance();

    hideElement(document.getElementById("checkinForm"));
    hideElement(document.getElementById("btnCheckin"));
    showElement(document.getElementById("checkoutForm"));
    showElement(document.getElementById("checkoutPreview"));
    hideElement(document.getElementById("checkoutEditForm"));
  });

  // Edit Checkout
  document.getElementById("editCheckout").addEventListener("click", () => {
    hideElement(document.getElementById("checkoutPreview"));
    showElement(document.getElementById("checkoutEditForm"));
    document.getElementById("currentPlaceOut").value = currentCheckin.currentPlace;
    document.getElementById("targetPlaceOut").value = currentCheckin.targetPlace;
    document.getElementById("reasonOut").value = currentCheckin.reason;
    document.getElementById("checkinTimeOut").textContent = new Date(currentCheckin.time).toLocaleString();
    document.getElementById("checkoutTimeOut").value = toDatetimeLocal(new Date());
    updatePreviewDistance();
  });

  // Save Preview Checkout
  document.getElementById("saveCheckoutPreview").addEventListener("click", () => {
    const checkinTime = currentCheckin.time;
    const checkoutTime = document.getElementById("checkinTimeOutPreview").value;
    if (!checkoutTime) {
      document.getElementById("error-checkinTimeOutPreview").textContent = "เลือกวันที่และเวลา";
      return;
    }
    saveCheckoutData({
      currentPlace: currentCheckin.currentPlace,
      targetPlace: currentCheckin.targetPlace,
      reason: currentCheckin.reason,
      checkinTime,
      checkoutTime,
      lat: currentCheckin.lat,
      lng: currentCheckin.lng,
      targetLat: currentCheckin.lat,
      targetLng: currentCheckin.lng
    });
  });

  // Save Edit Checkout
  document.getElementById("saveCheckoutEdit").addEventListener("click", () => {
    const currentPlace = document.getElementById("currentPlaceOut").value.trim();
    const targetPlace = document.getElementById("targetPlaceOut").value.trim();
    const reason = document.getElementById("reasonOut").value.trim();
    const checkinTime = currentCheckin.time;
    const checkoutTime = document.getElementById("checkoutTimeOut").value;
    saveCheckoutData({
      currentPlace,
      targetPlace,
      reason,
      checkinTime,
      checkoutTime,
      lat: currentCheckin.lat,
      lng: currentCheckin.lng,
      targetLat: currentCheckin.lat,
      targetLng: currentCheckin.lng
    });
  });

  // Modal close buttons
  ["closeModal", "modalConfirm"].forEach(id => {
    document.getElementById(id).addEventListener("click", () => {
      hideElement(document.getElementById("checkoutModal"));
      resetToInitialStep();
    });
  });

  ["closeLocationModal", "locationModalOk"].forEach(id => {
    document.getElementById(id).addEventListener("click", () => {
      hideElement(document.getElementById("locationModal"));
    });
  });

});

