<?php

namespace Modules\Frontend\Livewire\ListData;

use GuzzleHttp\Psr7\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Modules\Frontend\app\Http\Controllers\FrontendInitController;
use Modules\Frontend\Traits\WithPaginationLogic;
use Modules\Map\app\Models\ApprovDate;
use Modules\Map\app\Models\CheckInOut;
use Modules\Member\app\Models\Member;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Modules\Frontend\app\Emails\ApprovalNotification;
use Modules\Member\app\Http\Controllers\MemberController;

class ApprovalList extends Component
{
    use WithPaginationLogic;

    public $request_approval, $lists, $status,$member;
    public $selected_items = [];
    protected $listeners = ['confirm-from-bootstrap' => 'confirmAction'];
    public $confirm_action_type = null; // approve|reject|approve_selected|reject_selected
    public $confirm_code_req = null;
    public $confirm_open = false;
    public $tab_details_row = false;
    public $page_total, $cur_page = 1, $per_page = 10, $pages, $total_items;
    public $remarks = ''; // เหตุผลการปฏิเสธ
    public $show_reject_modal = false; // สำหรับ Bootstrap modal
    public $current_code_req = null; // เก็บ code_req ปัจจุบันสำหรับ reject modal
    public $approve_btn_list = false, $reject_btn_list = false;
    public $no_checkbox_selected = false; // ตรวจสอบว่ามีการเลือก checkbox หรือไม่
    
    public function mount()
    {
        // Initialize properties
        $this->show_reject_modal = false;
        $this->current_code_req = null;
        $this->remarks = '';
        $this->selected_items = [];
        $this->confirm_open = false;
        $this->tab_details_row = false;
        $this->no_checkbox_selected = false;
    }

    public function render()
    {
        $this->member = Auth::guard('member')->user();
        // กำหนด status พื้นฐานที่คนอนุมัติควรเห็น (2 สำหรับผู้อนุมัติคนที่ 1, 3 สำหรับผู้อนุมัติคนที่ 2)
        if (!empty($this->member)) {
            $hasFirst  = Member::where('approve_by_first_id', $this->member->id)->exists();
            $hasSecond = Member::where('approve_by_second_id', $this->member->id)->exists();
            switch ($hasFirst && !$hasSecond) {
                case true:
                    $this->status = [2,3,4,13];
                    break;
                case false:
                    $this->status = [3,4,13];
                    break;
            }
            // if ($hasFirst && !$hasSecond) 
            //     // เป็นเฉพาะผู้อนุมัติลำดับที่ 1
            //     $this->status = [2,3,4,13];
            //     break;
            //  elseif (!$hasFirst && $hasSecond) 
            //     // เป็นเฉพาะผู้อนุมัติลำดับที่ 2
            //     $this->status = [3,4,13];
            // elseif ($hasSecond) 
            //     // มีทั้งสองบทบาท ให้ถือว่าเป็นผู้อนุมัติลำดับที่ 2 เป็นหลัก
            //     $this->status = [3,4,13];
            // else 
            //     // ไม่ได้เป็นผู้อนุมัติให้ใครเลย ใช้ค่าเริ่มต้นเดิม (ถ้ามี) หรือ 2
            //     $this->status = $this->status ?: [2,3,4,13];
            
        }
        $this->request_approval = MemberController::CountItems($this->member->id);
        // if (!empty($this->request_approval)) {
            // Refactored to use a more robust grouping and prevent possible overwrites by using a compound key for lists.
            $this->lists = [];
            foreach ($this->getAllListItems() as $item) {
                $key = $item->code_req;
                if (!isset($this->lists[$key])) {
                    $this->lists[$key] = [
                        'name' => $item->member->name ?? '',
                        'department' => $item->member->department ?? '',
                        'submission_date' => $item->submission_date ? $item->submission_date->format('d/m/Y H:i') : '',
                        'status_all' => [],
                        'request_list' => [],
                        'member_dt' => $item->member,
                    ];
                }
                $this->lists[$key]['status_all'][] = $item->status;
                $this->lists[$key]['request_list'][] = [
                    'id' => $item->id,
                    'code_req' => $item->code_req,
                    'date' => $item->time ? $item->time->format('d/m/Y H:i') : '',
                    'status_request' => (!empty($item->status_request)) ? FrontendInitController::check_status_request(json_decode($item->status_request, true)) : '',
                    'current_place' => $item->current_place,
                    'target_place' => $item->target_place,
                    'reason' => $item->reason,
                    'distance_km' => $item->distance_km,
                    'oil_price' => $item->oil_price,
                    'time' => $item->time ? $item->time->format('H:i') : '',
                    'checkout_time' => $item->checkout_time ? $item->checkout_time->format('H:i') : '',
                    'checkout_lat' => $item->checkout_lat,
                    'checkout_lng' => $item->checkout_lng,
                    'checkout_address' => $item->checkout_address,
                    'checkout_distance_km' => $item->checkout_distance_km,
                    'checkout_oil_cost' => $item->checkout_oil_cost,
                    'status' => FrontendInitController::status_appove($item),
                    'status_id' => $item->status,
                    

                ];
            }
            foreach ($this->lists as $key => $list) {
                $min_status = !empty($list['status_all']) ? min($list['status_all']) : null;
                $this->lists[$key]['status'] = FrontendInitController::status_appove_list($min_status,$list['member_dt']);
            }
        // }
        return view('frontend::livewire.list-data.approval-list');
    }

    private function getAllListItems($status = [])
    {
        /**
         * ต้องการดึง "ข้อมูลรอการอนุมัติ" โดย:
         *  - ถ้าเราเป็นผู้อนุมัติคนที่ 1  -> ดูรายการของลูกน้องที่มีเราเป็น approve_by_first_id และ status = 2
         *  - ถ้าเราเป็นผู้อนุมัติคนที่ 2  -> ดูรายการของลูกน้องที่มีเราเป็น approve_by_second_id และ status = 3
         *  - ไม่สนใจ lavel_id
         */

        // หา member ที่เราเป็นคนอนุมัติลำดับที่ 1 หรือ 2
        $member_first_ids  = Member::where('approve_by_first_id', $this->member->id)->pluck('id')->toArray();
        $member_second_ids = Member::where('approve_by_second_id', $this->member->id)->pluck('id')->toArray();

        $member_ids = [];
        $statusFilter = [];

        // กำหนดบทบาทหลักของผู้ใช้ปัจจุบัน และสถานะที่ต้องดู
        $hasFirst  = !empty($member_first_ids);
        $hasSecond = !empty($member_second_ids);

        // Note: The original code used 'break;' outside of loop/switch, which is invalid in PHP
        // Fix: Simply use if/elseif/else block, no breaks

        if ($hasFirst && !$hasSecond) {
            // เป็นเฉพาะผู้อนุมัติลำดับที่ 1 -> ดู status = 2 ของลูกน้องที่เราเป็น approve_by_first_id
            $member_ids   = $member_first_ids;
            $statusFilter = [2,3,4,13];
        } elseif (!$hasFirst && $hasSecond) {
            // เป็นเฉพาะผู้อนุมัติลำดับที่ 2 -> ดู status = 3 ของลูกน้องที่เราเป็น approve_by_second_id
            $member_ids   = $member_second_ids;
            $statusFilter = [3,4,13];
        } elseif ($hasSecond) {
            // มีทั้งสองบทบาท ให้ถือว่าเป็นผู้อนุมัติลำดับที่ 2 เป็นหลัก
            $member_ids   = $member_second_ids;
            $statusFilter = [3,4,13];
        } else {
            // ไม่ได้เป็นผู้อนุมัติให้ใครเลย
            $this->total_items = 0;
            $this->page_total  = 0;
            $this->pages = [];
            return collect();
        }

        // unique member id
        $member_ids = array_values(array_unique($member_ids));

        $this->tab_details_row = true;

        // สำหรับผู้อนุมัติคนที่ 1: รวมรายการที่ตัวเองอนุมัติแล้ว (ตรวจสอบจาก ApprovDate)
        $approvedIds = [];
        if ($hasFirst && !$hasSecond) {
            // หา ID ของรายการที่ผู้อนุมัติคนที่ 1 อนุมัติแล้ว
            $approvedIds = ApprovDate::where('member_id', $this->member->id)
                ->pluck('approv_id')
                ->toArray();
            // Debug: ตรวจสอบว่า approvedIds มีข้อมูลหรือไม่
            Log::info('ApprovalList - First Approver Approved IDs', [
                'member_id' => $this->member->id,
                'approved_ids_count' => count($approvedIds),
                'approved_ids' => $approvedIds
            ]);
        }

        // สำหรับผู้อนุมัติคนที่ 1: รวมรายการที่ตัวเองอนุมัติแล้วด้วย
        // ใช้ whereIn แบบง่ายๆ โดยรวม approvedIds เข้าไปด้วย
        if ($hasFirst && !$hasSecond && !empty($approvedIds)) {
            // รวมรายการปกติ (member_ids + status) + รายการที่อนุมัติแล้ว (approvedIds)
            // ใช้ whereIn แยกเป็น 2 ส่วน: รายการปกติ + รายการที่อนุมัติแล้ว
            $query = CheckInOut::where(function($q) use ($member_ids, $statusFilter, $approvedIds) {
                $q->where(function($subQ) use ($member_ids, $statusFilter) {
                    $subQ->whereIn('member_id', $member_ids)
                         ->whereIn('status', $statusFilter);
                })->orWhereIn('id', $approvedIds);
            });
        } else {
            $query = CheckInOut::whereIn('member_id', $member_ids)
                ->whereIn('status', $statusFilter);
        }

        // นับจำนวน code_req ที่ไม่ซ้ำกัน
        $all = $query->distinct()->count('code_req');

        $this->total_items = $all;
        $this->page_total  = (int) ceil((int) $all / $this->per_page);
        $this->pages = setPagination($this->per_page, $this->page_total);

        // ดึง code_req ที่ไม่ซ้ำกันตาม pagination
        if ($hasFirst && !$hasSecond && !empty($approvedIds)) {
            $codeReqsQuery = CheckInOut::where(function($q) use ($member_ids, $statusFilter, $approvedIds) {
                $q->where(function($subQ) use ($member_ids, $statusFilter) {
                    $subQ->whereIn('member_id', $member_ids)
                         ->whereIn('status', $statusFilter);
                })->orWhereIn('id', $approvedIds);
            });
        } else {
            $codeReqsQuery = CheckInOut::whereIn('member_id', $member_ids)
                ->whereIn('status', $statusFilter);
        }
        
        $codeReqs = $codeReqsQuery->select('code_req')
            ->distinct()
            ->orderBy('code_req', 'desc')
            ->limit($this->per_page)
            ->offset(($this->cur_page - 1) * $this->per_page)
            ->pluck('code_req');

        // ดึงข้อมูลทั้งหมดของ code_req ที่ได้
        if ($hasFirst && !$hasSecond && !empty($approvedIds)) {
            $dataQuery = CheckInOut::where(function($q) use ($member_ids, $statusFilter, $approvedIds) {
                $q->where(function($subQ) use ($member_ids, $statusFilter) {
                    $subQ->whereIn('member_id', $member_ids)
                         ->whereIn('status', $statusFilter);
                })->orWhereIn('id', $approvedIds);
            });
        } else {
            $dataQuery = CheckInOut::whereIn('member_id', $member_ids)
                ->whereIn('status', $statusFilter);
        }
        
        $data = $dataQuery->whereIn('code_req', $codeReqs)
            ->orderBy('code_req', 'desc')
            ->orderBy('time', 'asc')
            ->get();
        return $data;
    }

    public function approve(string $codeReq)
    {
        if (empty($this->member)) {
            Log::warning('Approve - Member is empty');
            return;
        }

        // ดึงข้อมูลรายการทั้งหมดใน code_req
        $data_lists = CheckInOut::where('code_req', $codeReq)->get();
        
        if ($data_lists->isEmpty()) {
            Log::warning('Approve - No data found for code_req: ' . $codeReq);
            $this->dispatch('alert', [
                'type' => 'warning',
                'title' => 'ไม่พบรายการที่ต้องการอนุมัติ',
            ]);
            return;
        }

        // ดึงข้อมูล member ของผู้ขออนุมัติ (ใช้รายการแรกเป็นตัวแทน)
        $memberOfRequest = $data_lists->first()->member;
        if (empty($memberOfRequest)) {
            Log::warning('Approve - Member of request not found');
            return;
        }

        // ตรวจสอบว่าผู้ใช้ปัจจุบันเป็นผู้อนุมัติคนที่ 1 หรือ 2
        $isFirstApprover = ($memberOfRequest->approve_by_first_id == $this->member->id);
        $isSecondApprover = ($memberOfRequest->approve_by_second_id == $this->member->id);

        if (!$isFirstApprover && !$isSecondApprover) {
            Log::warning('Approve - Current user is not an approver for this request');
            $this->dispatch('alert', [
                'type' => 'warning',
                'title' => 'คุณไม่มีสิทธิ์อนุมัติรายการนี้',
            ]);
            return;
        }

        // ตรวจสอบว่ามีผู้อนุมัติกี่คน
        $hasFirstApprover = !empty($memberOfRequest->approve_by_first_id);
        $hasSecondApprover = !empty($memberOfRequest->approve_by_second_id);
        $numApprovers = ($hasFirstApprover ? 1 : 0) + ($hasSecondApprover ? 1 : 0);

        // กำหนดสถานะใหม่และสถานะที่สามารถอนุมัติได้
        $newStatus = null;
        $allowedStatuses = [];

        if ($numApprovers === 1) {
            // มีผู้อนุมัติคนเดียว -> อนุมัติแล้วเปลี่ยนเป็น 4 เลย
            $newStatus = 4;
            $allowedStatuses = [2]; // อนุมัติได้จากสถานะ 2
        } else {
            // มีผู้อนุมัติ 2 คน
            if ($isFirstApprover) {
                // เป็นผู้อนุมัติคนที่ 1 -> อนุมัติแล้วเปลี่ยนเป็น 3
                $newStatus = 3;
                $allowedStatuses = [2]; // อนุมัติได้จากสถานะ 2
            } elseif ($isSecondApprover) {
                // เป็นผู้อนุมัติคนที่ 2 -> ต้องตรวจสอบว่าคนที่ 1 อนุมัติแล้วหรือยัง
                // ดึง ID ของรายการทั้งหมดใน code_req
                $checkInOutIds = $data_lists->pluck('id')->toArray();
                
                // ตรวจสอบว่าผู้อนุมัติคนที่ 1 อนุมัติแล้วหรือยัง
                $firstApproverApproved = ApprovDate::whereIn('approv_id', $checkInOutIds)
                    ->where('member_id', $memberOfRequest->approve_by_first_id)
                    ->exists();
                
                if (!$firstApproverApproved) {
                    Log::warning('Approve - First approver has not approved yet');
                    $this->dispatch('alert', [
                        'type' => 'warning',
                        'title' => 'ผู้อนุมัติคนที่ 1 ยังไม่อนุมัติ',
                    ]);
                    return;
                }
                
                // อนุมัติแล้วเปลี่ยนเป็น 4
                $newStatus = 4;
                $allowedStatuses = [3]; // อนุมัติได้จากสถานะ 3 (หลังจากคนที่ 1 อนุมัติแล้ว)
            }
        }

        // Debug logging
        Log::info('Approve - Code Req: ' . $codeReq);
        Log::info('Approve - Current Member ID: ' . $this->member->id);
        Log::info('Approve - Is First Approver: ' . ($isFirstApprover ? 'Yes' : 'No'));
        Log::info('Approve - Is Second Approver: ' . ($isSecondApprover ? 'Yes' : 'No'));
        Log::info('Approve - Number of Approvers: ' . $numApprovers);
        Log::info('Approve - New Status: ' . $newStatus);
        Log::info('Approve - Allowed Statuses: ' . json_encode($allowedStatuses));
        
        // ดึง ID ของรายการที่จะอัปเดตก่อน
        $ids = CheckInOut::where('code_req', $codeReq)
            ->whereIn('status', $allowedStatuses)
            ->pluck('id')
            ->toArray();
        
        Log::info('Approve - Found IDs: ' . json_encode($ids));
        
        if (empty($ids)) {
            Log::warning('Approve - No items found with allowed statuses');
            $this->dispatch('alert', [
                'type' => 'warning',
                'title' => 'ไม่พบรายการที่สามารถอนุมัติได้',
            ]);
            return;
        }
        
        $affected = CheckInOut::where('code_req', $codeReq)
            ->whereIn('status', $allowedStatuses)
            ->update(['status' => $newStatus]);
        
        Log::info('Approve - Affected rows: ' . $affected);
            
        if (!empty($affected) && !empty($ids)) {
            // บันทึก ApprovDate สำหรับทุกรายการที่ได้รับการอนุมัติ
            foreach ($ids as $id) {
                // ตรวจสอบว่ามีการบันทึก ApprovDate สำหรับรายการนี้แล้วหรือยัง
                $existingApproval = ApprovDate::where('approv_id', $id)
                    ->where('member_id', $this->member->id)
                    ->first();
                
                if (!$existingApproval) {
                    ApprovDate::create([
                        'member_id' => $this->member->id,
                        'approv_id' => $id,
                        'date' => now(),
                    ]);
                }
            }
        }
        
        $this->selected_items = [];

        // ส่งอีเมลเมื่ออนุมัติสมบูรณ์ (สถานะ 4)
        if ($newStatus === 4) {
            $this->sendApprovalEmailNotifications($ids, $this->member, $newStatus);
        }
        
        $this->dispatch('alert', [
            'type' => $affected > 0 ? 'success' : 'warning',
            'title' => $affected > 0 ? 'ดำเนินการสำเร็จ' : 'ไม่พบรายการที่อัปเดตได้',
        ]);
    }

    public function reject(string $codeReq)
    {
        if (empty($this->member)) {
            return;
        }
        // กำหนดสถานะที่สามารถปฏิเสธได้ตามระดับผู้ใช้
        $allowedStatuses = ($this->member->lavel_id === 2) ? [2, 3] : [3, 4];
        
        // ตรวจสอบว่ามีรายการที่เลือกหรือไม่
        $selectedIds = array_values(array_unique(array_map('intval', (array) $this->selected_items)));
        
        // ถ้ามีรายการที่เลือก ให้ปฏิเสธเฉพาะที่เลือก
        // ถ้าไม่มีรายการที่เลือก ให้ปฏิเสธทั้งหมดใน code_req
        if (!empty($selectedIds)) {
            // ดึง ID ของรายการที่เลือกและอยู่ใน code_req ที่สามารถปฏิเสธได้
            $ids = CheckInOut::where('code_req', $codeReq)
                ->whereIn('id', $selectedIds)
                ->whereIn('status', $allowedStatuses)
                ->pluck('id')
                ->toArray();
        } else {
            // ดึง ID ของรายการทั้งหมดใน code_req ที่สามารถปฏิเสธได้
            $ids = CheckInOut::where('code_req', $codeReq)
                ->whereIn('status', $allowedStatuses)
                ->pluck('id')
                ->toArray();
        }
            
        if (empty($ids)) {
            $this->dispatch('alert', [
                'type' => 'warning',
                'title' => 'ไม่พบรายการที่สามารถปฏิเสธได้',
            ]);
            return;
        }
        // Debug logging
        Log::info('Reject - Code Req: ' . $codeReq);
        Log::info('Reject - Selected IDs: ' . json_encode($selectedIds));
        Log::info('Reject - Valid IDs: ' . json_encode($ids));
        Log::info('Reject - Remarks: ' . $this->remarks);
        Log::info('Reject - Allowed Statuses: ' . json_encode($allowedStatuses));
        
        $affected = CheckInOut::where('code_req', $codeReq)
            ->whereIn('id', $ids)
            ->whereIn('status', $allowedStatuses)
            ->update([
                'status' => 13,
                'remarks' => $this->remarks
            ]);

        Log::info('Reject - Affected rows: ' . $affected);

        if (!empty($affected) && !empty($ids)) {
            // บันทึก ApprovDate สำหรับทุกรายการที่ถูกปฏิเสธ
            foreach ($ids as $id) {
                ApprovDate::create([
                    'member_id' => $this->member->id,
                    'approv_id' => $id,
                    'date' => now(),
                ]);
            }
        }
        $this->selected_items = [];
        $this->remarks = ''; // รีเซ็ตค่า remarks

        $this->dispatch('alert', [
            'type' => $affected > 0 ? 'success' : 'warning',
            'title' => $affected > 0 ? 'ปฏิเสธรายการเรียบร้อย' : 'ไม่พบรายการที่อัปเดตได้',
        ]);
    }

    public function approveSelectedItems()
    {
        // sanitize selected IDs
        $ids = array_values(array_unique(array_map('intval', (array) $this->selected_items)));
        
        if (empty($this->member) || empty($ids)) {
            return;
        }

        // ดึงข้อมูลรายการตาม $ids ทั้งหมด พร้อม member relationship
        $items = CheckInOut::with('member')->whereIn('id', $ids)->get();

        $affected = 0;
        $newStatusById = []; // track id => newStatus (for email notification)

        foreach ($items as $item) {
            // ตรวจสอบว่ามี member หรือไม่
            if (empty($item->member)) {
                continue;
            }

            $memberOfRequest = $item->member;
            
            // ตรวจสอบว่าผู้ใช้ปัจจุบันเป็นผู้อนุมัติคนที่ 1 หรือ 2
            $isFirstApprover = ($memberOfRequest->approve_by_first_id == $this->member->id);
            $isSecondApprover = ($memberOfRequest->approve_by_second_id == $this->member->id);

            if (!$isFirstApprover && !$isSecondApprover) {
                // ไม่ใช่ผู้มีสิทธิ์อนุมัติ
                continue;
            }

            // ตรวจสอบว่ามีผู้อนุมัติกี่คน
            $hasFirstApprover = !empty($memberOfRequest->approve_by_first_id);
            $hasSecondApprover = !empty($memberOfRequest->approve_by_second_id);
            $numApprovers = ($hasFirstApprover ? 1 : 0) + ($hasSecondApprover ? 1 : 0);

            // กำหนดสถานะใหม่และสถานะที่สามารถอนุมัติได้
            $newStatus = null;
            $allowedStatuses = [];

            if ($numApprovers === 1) {
                // มีผู้อนุมัติคนเดียว -> อนุมัติแล้วเปลี่ยนเป็น 4 เลย
                $newStatus = 4;
                $allowedStatuses = [2];
            } else {
                // มีผู้อนุมัติ 2 คน
                if ($isFirstApprover) {
                    // เป็นผู้อนุมัติคนที่ 1 -> อนุมัติแล้วเปลี่ยนเป็น 3
                    $newStatus = 3;
                    $allowedStatuses = [2];
                } elseif ($isSecondApprover) {
                    // เป็นผู้อนุมัติคนที่ 2 -> ต้องตรวจสอบว่าคนที่ 1 อนุมัติแล้วหรือยัง
                    $firstApproverApproved = ApprovDate::where('approv_id', $item->id)
                        ->where('member_id', $memberOfRequest->approve_by_first_id)
                        ->exists();
                    
                    if (!$firstApproverApproved) {
                        // ผู้อนุมัติคนที่ 1 ยังไม่อนุมัติ
                        continue;
                    }
                    
                    // อนุมัติแล้วเปลี่ยนเป็น 4
                    $newStatus = 4;
                    $allowedStatuses = [3];
                }
            }

            // ตรวจสอบว่าสถานะปัจจุบันอยู่ใน allowedStatuses หรือไม่
            if (!in_array($item->status, $allowedStatuses)) {
                continue;
            }

            // อัปเดตสถานะรายการ
            $update = CheckInOut::where('id', $item->id)->update(['status' => $newStatus]);
            if ($update) {
                $affected++;
                $newStatusById[$item->id] = $newStatus;
                
                // ตรวจสอบว่ามีการบันทึก ApprovDate สำหรับรายการนี้แล้วหรือยัง
                $existingApproval = ApprovDate::where('approv_id', $item->id)
                    ->where('member_id', $this->member->id)
                    ->first();
                
                if (!$existingApproval) {
                    // บันทึกวันที่อนุมัติ
                    ApprovDate::create([
                        'member_id' => $this->member->id,
                        'approv_id' => $item->id,
                        'date' => now(),
                    ]);
                }
            }
        }

        // ถ้ามีสถานะที่อัปเดตเป็น 4 ให้ส่งอีเมลเฉพาะรายการเหล่านั้น
        $approvedIdsForEmail = array_keys(array_filter($newStatusById, function ($s) {
            return $s == 4;
        }));

        if (!empty($approvedIdsForEmail)) {
            $this->sendApprovalEmailNotifications($approvedIdsForEmail, $this->member, 4);
        }

        $this->selected_items = [];
        $this->dispatch('alert', [
            'type' => $affected > 0 ? 'success' : 'warning',
            'title' => $affected > 0 ? 'อนุมัติทั้งหมดที่เลือกเรียบร้อย' : 'ไม่พบรายการที่อัปเดตได้',
        ]);
    }

    public function rejectSelectedItems()
    {
        $ids = array_values(array_unique(array_map('intval', (array) $this->selected_items)));
        if (empty($this->member) || empty($ids)) {
            return;
        }

        // กำหนดสถานะที่สามารถปฏิเสธได้ตามระดับผู้ใช้
        $allowedStatuses = ($this->member->lavel_id === 2) ? [2, 3] : [3, 4];

        $affected = CheckInOut::whereIn('id', $ids)->whereIn('status', $allowedStatuses)->update([
            'status' => 13,
            'remarks' => $this->remarks
        ]);

        if (!empty($affected)) {
            // บันทึก ApprovDate สำหรับทุกรายการที่ถูกปฏิเสธ
            foreach ($ids as $id) {
                ApprovDate::create([
                    'member_id' => $this->member->id,
                    'approv_id' => $id,
                    'date' => now(),
                ]);
            }
        }

        $this->selected_items = [];
        $this->remarks = ''; // รีเซ็ตค่า remarks
        $this->dispatch('alert', [
            'type' => $affected > 0 ? 'success' : 'warning',
            'title' => $affected > 0 ? 'ไม่อนุมัติทั้งหมดที่เลือกเรียบร้อย' : 'ไม่พบรายการที่อัปเดตได้',
        ]);
    }

    // ===== Confirm flow handlers =====
    public function setApprove(string $codeReq)
    {
        $this->confirm_action_type = 'approve';
        $this->confirm_code_req = $codeReq;
        $this->confirm_open = true;
        $this->tab_details_row = true;
    }

    public function setReject(string $codeReq)
    {
        $this->confirm_action_type = 'reject';
        $this->confirm_code_req = $codeReq;
        $this->confirm_open = true;
    }

    public function setApproveSelected()
    {
        // ตรวจสอบว่ามีรายการที่เลือกหรือไม่
        $ids = array_values(array_unique(array_map('intval', (array) $this->selected_items)));
        if (empty($ids)) {
            $this->no_checkbox_selected = true;
            $this->confirm_action_type = 'approve_selected';
            $this->confirm_code_req = null;
            $this->confirm_open = true;
            return;
        }
        
        $this->no_checkbox_selected = false;
        $this->confirm_action_type = 'approve_selected';
        $this->confirm_code_req = null;
        $this->confirm_open = true;
    }

    public function setRejectSelected()
    {
        // ตรวจสอบว่ามีรายการที่เลือกหรือไม่
        $ids = array_values(array_unique(array_map('intval', (array) $this->selected_items)));
        if (empty($ids)) {
            $this->no_checkbox_selected = true;
            $this->confirm_action_type = 'reject_selected';
            $this->confirm_code_req = null;
            $this->confirm_open = true;
            return;
        }
        
        $this->no_checkbox_selected = false;
        $this->confirm_action_type = 'reject_selected';
        $this->confirm_code_req = null;
        $this->confirm_open = true;
    }

    public function confirmAction()
    {
        // ตรวจสอบว่ามีรายการที่เลือกหรือไม่สำหรับกรณี selected
        if (in_array($this->confirm_action_type, ['approve_selected', 'reject_selected'])) {
            $ids = array_values(array_unique(array_map('intval', (array) $this->selected_items)));
            if (empty($ids)) {
                $this->no_checkbox_selected = true;
                $this->dispatch('alert', [
                    'type' => 'warning',
                    'title' => 'กรุณาเลือกอย่างน้อย 1 รายการ',
                ]);
                return; // ไม่ปิด modal และไม่ดำเนินการต่อ
            }
            $this->no_checkbox_selected = false;
        }
        
        // ตรวจสอบ remarks ก่อนสำหรับการ reject
        if (in_array($this->confirm_action_type, ['reject', 'reject_selected'])) {
            Log::info('Validate Remarks - Type: ' . $this->confirm_action_type);
            Log::info('Validate Remarks - Value: "' . $this->remarks . '"');
            Log::info('Validate Remarks - Length: ' . strlen($this->remarks));
            
            if (empty(trim($this->remarks))) {
                $this->dispatch('alert', [
                    'type' => 'warning',
                    'title' => 'กรุณาระบุเหตุผลการปฏิเสธ',
                ]);
                return; // ไม่ปิด modal และไม่ดำเนินการต่อ
            }
        }
        // ปิดโมดัล เพื่อให้แน่ใจว่า overlay ถูกลบออกเสมอ
        $this->confirm_open = false;
        $this->no_checkbox_selected = false;
        $wasUpdated = false;
        if ($this->confirm_action_type === 'approve' && !empty($this->confirm_code_req)) {
            $this->approve($this->confirm_code_req);
            $wasUpdated = true;
        } elseif ($this->confirm_action_type === 'reject' && !empty($this->confirm_code_req)) {
            $this->reject($this->confirm_code_req);
            $wasUpdated = true;
        } elseif ($this->confirm_action_type === 'approve_selected') {
            $this->approveSelectedItems();
            $wasUpdated = true;
        } elseif ($this->confirm_action_type === 'reject_selected') {
            $this->rejectSelectedItems();
            $wasUpdated = true;
        }

        $this->confirm_action_type = null;
        $this->confirm_code_req = null;
        $this->tab_details_row = true;

        // รีเฟรชข้อมูลถ้ามีการอัปเดต
        if ($wasUpdated) {
            $this->request_approval = $this->getAllListItems();
        }
    }

    public function closeConfirm()
    {
        $this->confirm_open = false;
        $this->remarks = ''; // รีเซ็ตค่า remarks เมื่อปิด modal
        $this->no_checkbox_selected = false; // รีเซ็ตค่า no_checkbox_selected เมื่อปิด modal
    }

    public function toggleSingleItem($itemId, $checked)
    {
        $itemId = (int) $itemId;
        $checked = filter_var($checked, FILTER_VALIDATE_BOOLEAN);
        
        if ($checked) {
            // เพิ่ม ID ถ้ายังไม่มี
            if (!in_array($itemId, $this->selected_items)) {
                $this->selected_items[] = $itemId;
            }
            $this->approve_btn_list = true;
            $this->reject_btn_list = true;
        } else {
            // ลบ ID ออก
            $this->selected_items = array_values(array_filter($this->selected_items, function($id) use ($itemId) {
                return $id !== $itemId;
            }));
            
            // ถ้าไม่มีรายการที่เลือก ให้ปิดปุ่ม
            if (empty($this->selected_items)) {
                $this->approve_btn_list = false;
                $this->reject_btn_list = false;
            }
        }
    }

    public function toggleGroupSelection(string $codeReq, $checked)
    {
        // แปลง checked เป็น boolean
        $checked = filter_var($checked, FILTER_VALIDATE_BOOLEAN);
        if($checked) {
            $this->approve_btn_list = true;
            $this->reject_btn_list = true;
        }else{
            $this->approve_btn_list = false;
            $this->reject_btn_list = false;
        }
        if ($codeReq === 'ALL') {
            $allEligible = [];
            if (!empty($this->lists)) {
                foreach ($this->lists as $code => $list) {
                    if (empty($list['request_list'])) continue;
                    foreach ($list['request_list'] as $req) {
                        if (($req['status_id'] ?? null) === $this->status[0] && !empty($req['id'])) {
                            $allEligible[] = (int) $req['id'];
                        }
                    }
                }
            }
            if ($checked) {
                $this->selected_items = array_values(array_unique(array_merge(
                    array_map('intval', (array) $this->selected_items),
                    $allEligible
                )));
            } else {
                $remove = array_flip($allEligible);
                $this->selected_items = array_values(array_filter(array_map('intval', (array) $this->selected_items), function ($id) use ($remove) {
                    return !isset($remove[$id]);
                }));
            }
            return;
        }
        if (empty($this->lists[$codeReq]['request_list'])) {
            return;
        }
        $eligibleIds = [];
        foreach ($this->lists[$codeReq]['request_list'] as $req) {
            if (($req['status_id'] ?? null) === $this->status[0] && !empty($req['id'])) {
                $eligibleIds[] = (int) $req['id'];
            }
        }
        if ($checked) {
            $this->selected_items = array_values(array_unique(array_merge(
                array_map('intval', (array) $this->selected_items),
                $eligibleIds
            )));
        } else {
            $remain = [];
            $current = array_map('intval', (array) $this->selected_items);
            $remove = array_flip($eligibleIds);
            foreach ($current as $id) {
                if (!isset($remove[$id])) {
                    $remain[] = $id;
                }
            }
            $this->selected_items = $remain;
        }
    }

    // ===== Bootstrap Reject Modal handlers =====
    public function openRejectModal(string $codeReq)
    {
        $this->current_code_req = $codeReq;
        $this->show_reject_modal = true;
        $this->remarks = ''; // รีเซ็ต remarks
    }

    public function closeRejectModal()
    {
        $this->show_reject_modal = false;
        $this->current_code_req = null;
        $this->remarks = ''; // รีเซ็ต remarks
    }

    public function submitReject()
    {
        // ตรวจสอบว่ามี remarks หรือไม่
        if (empty(trim($this->remarks))) {
            $this->dispatch('alert', [
                'type' => 'warning',
                'title' => 'กรุณาระบุเหตุผลการปฏิเสธ',
            ]);
            return;
        }

        // ตรวจสอบว่ามี code_req หรือไม่
        if (empty($this->current_code_req)) {
            $this->dispatch('alert', [
                'type' => 'error',
                'title' => 'ไม่พบรหัสคำขอ',
            ]);
            return;
        }
        // เรียก method reject
        $this->reject($this->current_code_req);
        
        // ปิด modal
        $this->closeRejectModal();
    }

    /**
     * ส่งอีเมลแจ้งเตือนการอนุมัติ
     * ส่งไปยังผู้ขออนุมัติและผู้เกี่ยวข้อง
     */
    private function sendApprovalEmailNotifications($approvedIds, $approver, $newStatus)
    {
        try {
            // ตั้งค่า config mail
            config_mail();

            // ดึงข้อมูลรายการที่อนุมัติ
            $approvedItems = CheckInOut::with('member')
                ->whereIn('id', $approvedIds)
                ->get();

            if ($approvedItems->isEmpty()) {
                return;
            }

            // จัดกลุ่มตาม member_id เพื่อส่งอีเมลแยกตามคน
            $groupedByMember = $approvedItems->groupBy('member_id');

            foreach ($groupedByMember as $items) {
                $requestMember = $items->first()->member;
                
                // ตรวจสอบว่ามี email หรือไม่
                if (empty($requestMember->email)) {
                    continue;
                }

                // สร้างข้อมูลสำหรับอีเมล (ส่งเฉพาะเมื่อ BU อนุมัติสมบูรณ์แล้ว)
                $emailData = [
                    ['name' => '<strong>ผู้อนุมัติ:</strong>', 'value' => $approver->name ?? '-'],
                    ['name' => '<strong>รายการที่อนุมัติ:</strong>', 'value' => count($items) . ' รายการ'],
                    ['name' => '<strong>สถานะ:</strong>', 'value' => "<span style='color: #28a745;'>อนุมัติสมบูรณ์</span>"],
                    ['name' => '<strong>วันที่อนุมัติ:</strong>', 'value' => now()->format('d/m/Y H:i')],
                ];

                // เพิ่มรายละเอียดของแต่ละรายการ
                foreach ($items as $index => $item) {
                    $emailData[] = [
                        'name' => "<strong>รายการที่ " . ($index + 1) . ":</strong>",
                        'value' => ''
                    ];
                    $emailData[] = [
                        'name' => '&nbsp;&nbsp;สถานที่ปัจจุบัน:',
                        'value' => $item->current_place ?? '-'
                    ];
                    $emailData[] = [
                        'name' => '&nbsp;&nbsp;สถานที่ไป:',
                        'value' => $item->target_place ?? '-'
                    ];
                    $emailData[] = [
                        'name' => '&nbsp;&nbsp;ระยะทาง:',
                        'value' => ($item->distance_km ?? 0) . ' กม.'
                    ];
                    $emailData[] = [
                        'name' => '&nbsp;&nbsp;วันที่:',
                        'value' => $item->time ? $item->time->format('d/m/Y H:i') : '-'
                    ];
                }

                // ส่งอีเมลไปยังผู้ขออนุมัติ
                Mail::to($requestMember->email)->send(
                    new ApprovalNotification(
                        'แจ้งเตือน: คำขออนุมัติของคุณได้รับการอนุมัติแล้ว',
                        $emailData
                    )
                );

                // ส่งอีเมลไปยังผู้บังคับบัญชาถ้ามี
                if (!empty($requestMember->parent_id)) {
                    $supervisor = Member::find($requestMember->parent_id);
                    if (!empty($supervisor->email)) {
                        $supervisorEmailData = array_merge(
                            [['name' => '<strong>ผู้ขออนุมัติ:</strong>', 'value' => $requestMember->name ?? '-']],
                            $emailData
                        );

                        Mail::to($supervisor->email)->send(
                            new ApprovalNotification(
                                'แจ้งเตือน:คำขออนุมัติของพนักงานได้รับการอนุมัติแล้ว',
                                $supervisorEmailData
                            )
                        );
                    }
                }
            }

            Log::info('Approval email notifications sent successfully', [
                'approved_ids' => $approvedIds,
                'approver_id' => $approver->id,
            ]);

        } catch (\Exception $e) {
            // Log error แต่ไม่ throw exception เพื่อไม่ให้กระทบกับการอนุมัติ
            Log::error('Failed to send approval email notifications', [
                'error' => $e->getMessage(),
                'approved_ids' => $approvedIds,
                'approver_id' => $approver->id ?? null,
            ]);
        }
    }
}
