<?php

namespace Modules\Frontend\Livewire\Form;

use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\On;
use Livewire\Component;
use Modules\Map\app\Models\CheckInOut;
use Modules\Member\app\Models\Member;
use Modules\Member\app\Models\MemberLavel;
use Modules\OilPrice\app\Models\OilPrice;

class CheckoutMap extends Component
{
    public $checkin_id = '',$type = '';
    public $checkin = null, $member, $checkoutEditForm = false;
    
    // Editable fields
    public $current_place = '';
    public $target_place = '';
    public $reason = '';
    
    // Checkout location
    public $checkout_lat = null;
    public $checkout_lng = null;
    public $checkout_address = '';
    public $distance_km = null;
    
    // Location names from coordinates
    public $checkin_location_name = '';
    public $checkout_location_name = '';
    
    // Modal state
    public $showDistanceModal = false;
    public $distanceLoading = false;
    public $distanceError = false;
    public $errorMessage = '';

    public function mount()
    {
        $this->checkin_id = session('checkin_id');
    }

    public function render()
    {
        $this->checkin = CheckInOut::find($this->checkin_id);
        $this->member = Member::find($this->checkin->member_id);
        return view('frontend::livewire.form.checkout-map');
    }

    public function editCheckout()
    {
        $this->checkoutEditForm = true;
        
        // Load current values into editable properties
        if ($this->checkin) {
            $this->current_place = $this->checkin->current_place ?? '';
            $this->target_place = $this->checkin->target_place ?? '';
            $this->reason = $this->checkin->reason ?? '';
        }
    }

    public function cancelEditCheckout()
    {
        $this->checkoutEditForm = false;
        
        // Reset form fields
        $this->current_place = '';
        $this->target_place = '';
        $this->reason = '';
    }

    public function saveCheckoutEdit()
    {
        // Validate input
        $this->validate([
            'current_place' => 'required|string|max:255',
            'target_place' => 'required|string|max:255',
            'reason' => 'required|string|max:255',
            'checkout_lat' => 'required|numeric',
            'checkout_lng' => 'required|numeric',
        ], [
            'current_place.required' => 'กรุณากรอกชื่อสถานที่ปัจจุบัน',
            'target_place.required' => 'กรุณากรอกชื่อสถานที่จะไป',
            'reason.required' => 'กรุณากรอกเหตุผล',
            'checkout_lat.required' => 'กรุณาอนุญาตการเข้าถึงตำแหน่งของคุณ',
            'checkout_lng.required' => 'กรุณาอนุญาตการเข้าถึงตำแหน่งของคุณ',
        ]);
        
        // Calculate distance using Google Maps Distance Matrix API
        $this->calculateDistance();
    }
    
    public function saveCheckoutPreview()
    {

       
        // Use data from checkin (no edit)
        $checkin = CheckInOut::find($this->checkin_id);
        if (!$checkin) {
            session()->flash('error', 'ไม่พบข้อมูล Check-in');
            return;
        }
        
        // Set data from existing checkin
        $this->current_place = $checkin->current_place;
        $this->target_place = $checkin->target_place;
        $this->reason = $checkin->reason;
        
        // Check if location is available
        if (!$this->checkout_lat || !$this->checkout_lng) {
            // Dispatch event to get location from JavaScript
            $this->dispatch('get-checkout-location-and-save');
            return;
        }
        
        // Calculate distance using Google Maps Distance Matrix API
        $this->calculateDistance();
    }
    
    public function calculateDistance()
    {
        $checkin = CheckInOut::find($this->checkin_id);
        
        if (!$checkin) {
            $this->distanceError = true;
            $this->errorMessage = 'ไม่พบข้อมูล Check-in';
            return;
        }
        
        $apiKey = env('GOOGLE_MAPS_API_KEY', 'AIzaSyCsTYM_vFrL0PuZ9cCVH3iAv3eWvsQmmpU');
        $origin = $checkin->lat . ',' . $checkin->lng;
        $destination = $this->checkout_lat . ',' . $this->checkout_lng;
        
        try {
            $url = "https://maps.googleapis.com/maps/api/distancematrix/json?origins={$origin}&destinations={$destination}&key={$apiKey}&language=th";
            
            $response = file_get_contents($url);
            $data = json_decode($response, true);
            
            if ($data['status'] === 'OK' && isset($data['rows'][0]['elements'][0]['distance'])) {
                $distanceMeters = $data['rows'][0]['elements'][0]['distance']['value'];
                $this->distance_km = round($distanceMeters / 1000, 2);
                
                // Show modal with distance summary
                $this->showDistanceModal = true;
                $this->distanceLoading = false;
            } else {
                $this->distanceError = true;
                $this->errorMessage = 'ไม่สามารถคำนวณระยะทางได้';
            }
        } catch (\Exception $e) {
            $this->distanceError = true;
            $this->errorMessage = 'เกิดข้อผิดพลาด: ' . $e->getMessage();
        }
    }
    
    public function confirmSaveCheckout()
    {
        // Update check-in record in database
        $checkin = CheckInOut::find($this->checkin_id);
        
        if ($checkin) {
            $role = Auth::guard('member')->user();
            $scheme_type = '';
            $lavel = MemberLavel::find($role->lavel_id);
            $oil_price = OilPrice::where([['status', 1]])->where('scheme_type', $lavel->id)->first();
            if (!empty($oil_price->rate)) {
                $oil_price_amount = $oil_price->rate * $this->distance_km;
            }else{
                $oil_price_amount = 0;
            }
            $checkin->update([
                'code_req' => $this->ger_code($checkin->member_id),
                'submission_date' => now(),
                'current_place' => $this->current_place,
                'target_place' => $this->target_place,
                'reason' => $this->reason,
                'checkout_lat' => $this->checkout_lat,
                'checkout_lng' => $this->checkout_lng,
                'checkout_time' => now(),
                'checkout_address' => $this->checkout_address,
                'distance_km' => $this->distance_km,
                'oil_price' => $oil_price_amount,
                'status' => $checkin->status + 1,
            ]);

            // Close modal and edit form
            $this->showDistanceModal = false;
            $this->checkoutEditForm = false;
            
            // Reset checkout data
            $this->resetCheckoutData();
            
            // เคลียร์ session ที่เกี่ยวข้องกับการเช็คอิน
            session()->forget(['checkin_id', 'checkin_id_expires_at']);
            
            // Show success message and redirect
            session()->flash('success', 'บันทึกข้อมูล Check-out สำเร็จ');
            
            // Redirect to personalinformation page
            return redirect()->route('frontend.check');
        } else {
            session()->flash('error', 'ไม่พบข้อมูล Check-in');
        }
    }

    private function ger_code($member_id){
         // เงื่อนไข: ให้มี member_id และ date 4 หลัก
        // code_req เดิม: REQ-memberid-YYYYMMDD-#### 
        // ใหม่: code_req = REQ-memberid-YYYYMMDD-HHMMSS-####
        $today = date('Y-m-d');
        $now = now();
        $currentTime = $now->format('His'); // HHMMSS

        $lastRequest = CheckInOut::where('member_id', $member_id)
            ->whereDate('created_at', $today)
            ->whereNotNull('code_req')
            // หาอันล่าสุดของวันนี้ที่มี HHMMSS เดียวกัน
            ->where('code_req', 'like', sprintf('REQ-%s-%s-%s-%%%%', $member_id, date('Ymd'), $currentTime))
            ->orderByDesc('created_at')
            ->first();

        $nextSeq = 1;
        if ($lastRequest && preg_match('/REQ-\d+-\d+-\d+-([0-9]{4})/', $lastRequest->code_req, $matches)) {
            $nextSeq = intval($matches[1]) + 1;
        }

        $code_req = sprintf(
            'REQ-%s-%s-%s-%04d',
            $member_id,
            date('Ymd'),
            $currentTime,
            $nextSeq
        );
        return $code_req;
    }
    
    public function closeDistanceModal()
    {
        $this->showDistanceModal = false;
        $this->distanceError = false;
        $this->distanceLoading = false;
    }
    
    public function resetCheckoutData()
    {
        $this->checkout_lat = null;
        $this->checkout_lng = null;
        $this->checkout_address = '';
        $this->distance_km = null;
    }
}
