<?php

namespace Modules\Contact\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Modules\Contact\app\Models\Subjects;
use Modules\Contact\app\Models\SubjectTranslate;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Http\Controllers\SlugController;

class SubjectsAdminController extends AdminController
{
	private $config, $action;

	/**
	 * Function : construct
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{
		$routeName = Route::currentRouteName();
		if ($routeName) {
			$explode = explode('.', $routeName);
			$this->action = $explode[3] ?? '';
		} else {
			$this->action = '';
		}

		self::$navbar = [
			['name' => __("contact::subject.title"), 'url' => null],
			['name' => __("contact::subject.name"), 'url' => route_has("admin.contact.subject.index")]
		];

		$this->config = config('contact.subject');
	}

	/**
	 * Function : subject index
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @return \Illuminate\Contracts\View\View contact::subject.index
	 */
	public function index()
	{
		return view('contact::subject.index', ['navbar' => self::$navbar, 'config' => $this->config]);
	}

	/**
	 * Function : subject datatable ajax
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return Yajra\DataTables\Facades\DataTables data table
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['sequence'];
			if ($this->config['upload']['status']) {
				array_push($dt_column, 'image');
			}
			if ($this->config['input']['name']['status']) {
				array_push($dt_column, 'name');
			}
			array_push($dt_column, 'sort', 'updated_at', 'action');
			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create subject object
			$req = new Subjects();

			// dt_search 
			if (!empty($dt_search)) {
				$req = $req->whereRelation('local', 'name', 'LIKE', "%{$dt_search}%");
			}

			// count all subject
			$dt_total = $req->count();
			// set query order & limit from datatable
			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)
				->limit($dt_length)
				->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('menu_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('image', function ($record) {
					return $this->view_file('image', $record->image);
				})->editColumn('name', function ($record) {
					return $record->local->name ?? '';
				})->editColumn('sort', function ($record) use ($dt_order) {
					return self::btn_sort("admin.contact.subject.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					if (!empty($this->config['btn']['status']))
						$btn .= self::btn_status("admin.contact.subject.set_status", $record->id, 'setStatus', $record->status);
					if (!empty($this->config['btn']['edit']))
						$btn .= self::btn_route("admin.contact.subject.edit", $record->id);
					if (!empty($this->config['btn']['delete']))
						$btn .= self::btn_delete("admin.contact.subject.set_delete", $record->id, 'setDelete');
					$btn .= '</div>';
					return $btn;
				})->escapeColumns([]);
			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : subject form
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param int $id
	 * @return \Illuminate\Contracts\View\View contact::subject.form
	 */
	public function form($id = 0)
	{
		$data = [];
		self::$navbar[] =  ['name' => __("action.{$this->action}"), 'url' => null];
		$meta = self::set_metadata('contact', 'subject', 1);

		if (!empty($id)) {
			$data = Subjects::find($id);
			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}
			$meta = array_merge($meta, SlugController::getMetadata('contact', 'subject', $id));
		}

		return view('contact::subject.form', [
			'data' => $data,
			'navbar' => self::$navbar,
			'meta' => $meta,
			'config' => $this->config,
			'action' => $this->action
		]);
	}

	/**
	 * Function : subject save
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function save(Request $request)
	{

		$rule = $msg = [];
		foreach (languages() as $lang => $l) {
			foreach ($this->config['input'] as $key => $item) {
				if (!empty($item['status']) && !empty($item['validate'])) {
					$rule["{$key}_{$lang}"] = ($key != 'name') ? 'required' : 'required|max:255';
					$msg["{$key}_{$lang}.*"] = __("field.{$key}_placeholder") . ' ' . ($l['name'] ?? '');
				}
			}
		}
		//validate post data
		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails()) {
			return self::response(['msg' => $valid->errors()->first()], Response::HTTP_MOVED_PERMANENTLY);
		}

		$attr['status'] = $request->get('status') ?? 0;

		if (check_file_upload($request, 'image', 'image')) {
			return self::response(['msg' => "Image " . __('app.resp.image')], Response::HTTP_BAD_REQUEST);
		} elseif ($this->config['upload']['image']) {
			$attr['image'] = self::set_image_upload($request, 'image', "public/contact", "image-");
		}
		if (empty($request->get('id'))) {
			$attr["sequence"] = Subjects::max('sequence') + 1;
		}

		$set = Subjects::updateOrcreate(['id' => $request->get('id')], $attr);
		if ($set->save()) {
			self::setLangs($request, $set);
			if ($this->config['seo']) {
				SlugController::createMetadata($request, $set->is);
			}
			return self::response(['msg' => __('noti.success')]);
		}
		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : subject set translate
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $id
	 * @return void
	 */
	private function setLangs(Request $request, $row)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($this->config['input'] as $key => $item) {
				$attr[$key] = mwz_setTextString($request->get("{$key}_{$lang}") ?? '', $item['type'] == 'texteditor');
			}
			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}

	/**
	 * Function : subject set status
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_status(Request $request)
	{
		$set = Subjects::find($request->get('id'));
		$set->status = !$set->status;
		if ($this->config['seo']) {
			SlugController::slug_status('contact', 'subject', $set->id, $set->status);
		}
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : subject set delete
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_delete(Request $request)
	{
		$set = Subjects::find($request->get('id'));
		// delete image
		if (!empty($set->first()->image))
			$this->delete_image($set->first()->image);
		// delete row
		if ($set->delete()) {
			if ($this->config['seo']) {
				SlugController::remove_metadata('contact', 'subject', $request->get('id'));
			}
			SubjectTranslate::where('subject_id', $request->get('id'))->delete();
			self::re_order();
			return self::response(['msg' => __('noti.delete')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : subject set re order
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_re_order(Request $request)
	{
		$sort_json = json_decode($request->get('sort_json'), 1);
		if (!empty($sort_json)) {
			foreach ($sort_json as $id => $sort) {
				Subjects::find($id)->update(['sequence' => $sort]);
			}
			self::re_order();
			return self::response(['msg' => __('noti.order')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : subject set sort
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_sort(Request $request)
	{
		$node = Subjects::find($request->get('id'));
		$move = $request->get('move') ?? 'up';
		$sort = $move == 'up' ? $node->decrement('sequence') : $node->increment('sequence');
		if ($sort) {
			self::re_order($move == 'up' ? 'DESC' : 'ASC');
			return self::response(['msg' =>  __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : subject re order
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @return void
	 */
	public function re_order($dir = 'DESC')
	{
		$list = Subjects::orderBy('sequence')->orderBy('updated_at', $dir)->get();
		foreach ($list as $cnt => $cat) {
			$cat->sequence = $cnt + 1;
			$cat->save();
		}
	}
}
