<?php

namespace Modules\Contact\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Log;
use Modules\Contact\app\Models\ContactInfo;
use Modules\Contact\app\Models\ContactInfoTranslate;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Http\Controllers\SlugController;

class ContactInfoAdminController extends AdminController
{
	private  $config, $slug;

	/**
	 * Function : construct
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{
		self::$navbar[] =  ['name' => __("contact::info.title"), 'url' => null];
		self::$navbar[] =  ['name' => __("contact::info.name"), 'url' => route_has("admin.contact.info.index")];

		$this->config = config('contact.info');

		$this->slug = new SlugController();
	}

	/**
	 * Function : contact info form
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @return \Illuminate\Contracts\View\View contact::info.form
	 */
	public function index()
	{
		$meta = $this->set_metadata('contact', 'contact', 1);
		$data = [];
		$data = ContactInfo::find(1) ?? '';
		if (!empty($data)) {
			if (!empty($data->social)) {
				$data->social = json_decode($data->social ?? null, JSON_UNESCAPED_UNICODE);
			}
			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}
		}

		if (!empty($this->config['seo'])) {
			$meta = array_merge($meta, SlugController::getMetadata('contact', 'contact', 1));
		}
		return view('contact::info.form', ['data' => $data, 'navbar' => self::$navbar, 'meta' => $meta, 'config' => $this->config]);
	}

	/**
	 * Function : contact save info
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function save(Request $request)
	{
		try {
			$rule = $msg = [];
			foreach ($this->config['input'] as $key => $item) {
				if (!empty($item['status']) && !empty($item['validate'])) {
					foreach (languages() as $lang => $l) {
						$maxLength = ($key === 'name') ? 500 : 255;
						$rule["{$key}_{$lang}"] = "required|max:{$maxLength}";
						$msg["{$key}_{$lang}.required"] = "Please specify " . __("field.{$key}_placeholder") . ' ' . ($l['name'] ?? '');
						$msg["{$key}_{$lang}.max"] = "Please specify " . __("field.{$key}_placeholder") . ' must not exceed ' . $maxLength . ' characters ' . ($l['name'] ?? '');
					}
				}
			}

			foreach ($this->config['contact'] as $key => $item) {
				if (!empty($item['status']) && !empty($item['validate']) || in_array($key, ['email', 'tel'])) {
					$max = 255;
					if ($key == 'email') {
						if (!str_contains($request->get('email'), '@') and !empty($request->get('email'))) {
							return self::response(['msg' => 'Invalid data'], Response::HTTP_MOVED_PERMANENTLY);
						}
						if (!empty($item['validate'])) {
							$rule[$key] = "required|email|max:{$max}";
							$msg["{$key}.regex"] = "Please specify " . __("field.email_placeholder") . ' is invalid';
						}
					} elseif ($key == 'tel' && !empty($item['validate'])) {
						$rule[$key] = "required|max:{$max}";
					} elseif (!empty($item['maxlength'])) {
						$max = $item['maxlength'];
					}

					$msg["{$key}.required"] = "Please specify " . __("field.{$key}_placeholder");
					$msg["{$key}.max"] = "Please specify " . __("field.{$key}_placeholder") . ' must not exceed ' . $max . ' characters';
				}
			}
			//validate post data
			$valid = validator($request->all(), $rule, $msg);
			if ($valid->fails()) {
				return self::response(['msg' => 'Invalid data', 'error' => $valid->errors()], Response::HTTP_MOVED_PERMANENTLY);
			}

			$attr = [
				'tel' => $request->get('tel'),
				'email' => strtolower($request->get('email')), // Convert email to lowercase
				'google_map' => mwz_setTextString($request->get("gmap")),
			];
			// upload image
			if (check_file_upload($request, 'image', 'image')) {
				return self::response(['msg' => "Image " . __('validation.attributes.error.image.mimetypes')], Response::HTTP_NOT_FOUND);
			} elseif ($this->config['upload']['image']) {
				$attr['image'] =  self::set_image_upload($request, 'image', "public/contact", "image-");
			}
			$attr['social'] = json_encode($request->get('social') ?? null, JSON_UNESCAPED_UNICODE);

			$set = ContactInfo::updateOrcreate(['id' => 1], $attr);
			if ($set->save()) {
				self::setLangs($request, $set);
				if ($this->config['seo']) {
					SlugController::createMetadata($request, 1);
				}
				return self::response(['msg' => __('noti.success')]);
			}
			return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
		} catch (\Exception $e) {
			// Check for UTF-8 related errors
			if (
				strpos($e->getMessage(), 'Malformed UTF-8') !== false ||
				strpos($e->getMessage(), 'String data, right truncated') !== false
			) {
				return self::response(['msg' => 'Invalid data'], Response::HTTP_MOVED_PERMANENTLY);
			}

			// Log the error for debugging
			Log::error('Contact info save error: ' . $e->getMessage());
			return self::response(['msg' => 'Invalid data'], Response::HTTP_MOVED_PERMANENTLY);
		}
	}

	/**
	 * Function : contact set translate
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $id
	 * @return void
	 */
	private function setLangs(Request $request, $row)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($this->config['input'] as $key => $item) {
				$value = $request->get("{$key}_{$lang}") ?? '';

				// Sanitize UTF-8 encoding
				if (!empty($value)) {
					$value = mwz_sanitizeUtf8($value);
				}

				// Log the original value for debugging
				if ($key === 'name' && !empty($value)) {
					Log::info("Contact info name field - Lang: {$lang}, Original length: " . strlen($value));
					if (strlen($value) > 100) {
						Log::info("Contact info name field - First 100 chars: " . substr($value, 0, 100));
					}
				}

				// Ensure the value doesn't exceed the field length limit
				$maxLength = ($key === 'name') ? 500 : 255;
				if (strlen($value) > $maxLength) {
					$originalValue = $value;
					$value = substr($value, 0, $maxLength);
					Log::warning("Contact info {$key} field truncated - Lang: {$lang}, Original length: " . strlen($originalValue) . ", New length: " . strlen($value));
				}

				$attr[$key] = mwz_setTextString($value, $item['type'] == 'texteditor');
			}
			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}
}
