<?php

namespace Modules\Contact\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Modules\Contact\app\Models\Branchs;
use Modules\Contact\app\Models\BranchTranslate;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Http\Controllers\SlugController;

class BranchAdminController extends AdminController
{
	private $module, $method, $action, $config;

	/**
	 * Function : construct
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{
		$this->middleware('auth:admin');

		$routeName = Route::currentRouteName();
		if ($routeName) {
			$explode = explode('.', $routeName);
			$this->module = $explode[1] ?? '';
			$this->method = $explode[2] ?? '';
			$this->action = $explode[3] ?? '';
		} else {
			$this->module = '';
			$this->method = '';
			$this->action = '';
		}

		self::$navbar = [
			['name' => __("contact::branch.{$this->method}.title"), 'url' => null],
			['name' => __("contact::branch.{$this->method}.name"), 'url' => route_has("admin.branch.{$this->method}.index")]
		];
		
		$this->config = config('contact.'.$this->method);
	}

	/**
	 * Function : branch index
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @return \Illuminate\Contracts\View\View contact::branch.index
	 */
	public function index()
	{
		return view('contact::branch.index', ['navbar' => self::$navbar, 'config' => $this->config,'type' => $this->method]);
	}

	/**
	 * Function : branch datatable ajax
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return Yajra\DataTables\Facades\DataTables data table
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['sequence'];
			if ($this->config['upload']['status']) {
				array_push($dt_column, 'image');
			}
			array_push($dt_column, 'name', 'sort', 'updated_at', 'action');
			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create branch object
			$req = new Branchs();
			// dt_search 
			if (!empty($dt_search)) {
				$req = $req->where(function ($query) use ($dt_search) {
					$query->whereRelation('local', 'name', 'like', "%$dt_search%");
				});
			}
			$req = $req->where('type', $this->method);

			// count all branch
			$dt_total = $req->count();
			// set query order & limit from datatable
			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)
				->limit($dt_length)
				->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('menu_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('image', function ($record) {
					return $this->view_file('image', $record->image);
				})->editColumn('name', function ($record) {
					return $record->local->name ?? '';
				})->editColumn('sort', function ($record) use ($dt_order) {
					return self::btn_sort("admin.contact.".$this->method.".set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					if (!empty($this->config['btn']['status']))
						$btn .= self::btn_status("admin.contact.".$this->method.".set_status", $record->id, 'setStatus', $record->status);
					if (!empty($this->config['btn']['edit']))
						$btn .= self::btn_route("admin.contact.".$this->method.".edit", $record->id);
					if (!empty($this->config['btn']['delete']))
						$btn .= self::btn_delete("admin.contact.".$this->method.".set_delete", $record->id, 'setDelete');
					$btn .= '</div>';
					return $btn;
				})->escapeColumns([]);

			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : branch form
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param int $id
	 * @return \Illuminate\Contracts\View\View contact::branch.form
	 */
	public function form($id = 0)
	{
		$data = [];
		self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
		$meta = $this->set_metadata('contact', 'branch', 1);
		if (!empty($id)) {
			$data = Branchs::find($id);
			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}
			$meta = array_merge($meta, SlugController::getMetadata('contact', 'branch', $id));
			if($this->config['contact']['tel']['multiple']) $data->tel = json_decode($data->tel,true);
            if($this->config['contact']['email']['multiple']) $data->email = json_decode($data->email,true);
		}

		return view('contact::branch.form', ['data' => $data, 'navbar' => self::$navbar, 'meta' => $meta, 'config' => $this->config,'type' => $this->method,'module' => $this->module]);
	}

	/**
	 * Function : branch save
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function save(Request $request)
	{
		$rule = $msg = [];
		foreach (languages() as $lang => $l) {
			foreach ($this->config['input'] as $key => $item) {
				if (!empty($item['status']) && !empty($item['validate'])) {
					$rule["{$key}_{$lang}"] = ($key != 'name') ? 'required' : 'required|max:255';
					$msg["{$key}_{$lang}.*"] = __("field.{$key}_placeholder") . ' ' . ($l['name'] ?? '');
				}
			}
		}
		//validate post data
		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first()], Response::HTTP_MOVED_PERMANENTLY);

		$attr = [
			'type' => $this->method,
			'tel' => ($this->config['contact']['tel']['multiple']) ? json_encode($request->get('tel'),true) : $request->get('tel'),
            'email' => ($this->config['contact']['email']['multiple']) ? json_encode($request->get('email'),true) : $request->get('email'),
			'google_map' => mwz_setTextString($request->get("gmap")),
			'status' => $request->get('status') ?? 0,
		];

		foreach ($this->config['social'] as $k => $v) {
			if ($v['status'])
				$attr[$k] = json_encode($request->get($k), 1);
		}
		// upload image
		if (check_file_upload($request, 'image', 'image')) {
			return self::response(['msg' => "Image " . __('app.resp.image')], Response::HTTP_BAD_REQUEST);
		} elseif ($this->config['upload']['image']) {
			$attr['image'] =  self::set_image_upload($request, 'image', "public/contact", "image-");
		}

		if (empty($request->get('id'))) {
			$attr["sequence"] = Branchs::max('sequence') + 1;
		}

		$set = Branchs::updateOrcreate(['id' => $request->get('id')], $attr);
		if ($set->save()) {
			self::setLangs($request, $set);
			if ($this->config['seo'])
				SlugController::createMetadata($request, $set->id);
			return self::response(['msg' => __('noti.success')]);
		}
		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : branch set translate
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $id
	 * @return void
	 */
	private function setLangs(Request $request, $row)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($this->config['input'] as $key => $item) {
				$attr[$key] = mwz_setTextString($request->get("{$key}_{$lang}") ?? '', $item['type'] == 'texteditor');
			}
			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}
	/**
	 * Function : branch set status
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_status(Request $request)
	{
		$set = Branchs::find($request->get('id'));
		$set->status = !$set->status;
		if ($this->config['seo']) {
			SlugController::slug_status('contact', 'branch', $set->id, $set->status);
		}
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : branch set delete
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_delete(Request $request)
	{
		$set = Branchs::find($request->get('id'));
		// delete image
		if (!empty($set->first()->image))
			$this->delete_image($set->first()->image);
		// delete row
		if ($set->delete()) {
			if ($this->config['seo']) {
				SlugController::remove_metadata('contact', 'branch', 1);
			}
			BranchTranslate::where('branch_id', $request->get('id'))->delete();
			$this->re_order();
			return self::response(['msg' => __('noti.delete')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : branch re order
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_re_order(Request $request)
	{
		$sort_json = json_decode($request->get('sort_json'), 1);
		if (!empty($sort_json)) {
			foreach ($sort_json as $id => $sort) {
				Branchs::find($id)->update(['sequence' => $sort]);
			}
			$this->re_order();

			return self::response(['msg' => __('noti.order')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : branch set sort
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_sort(Request $request)
	{
		$node = Branchs::find($request->get('id'));
		$move = $request->get('move') ?? 'up';
		$sort = $move == 'up' ? $node->decrement('sequence') : $node->increment('sequence');
		if ($sort) {
			self::re_order($move == 'up' ? 'DESC' : 'ASC');
			return self::response(['msg' =>  __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : branch re order
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @return void
	 */
	public function re_order($dir = 'DESC')
	{
		$list = Branchs::orderBy('sequence')->orderBy('updated_at', $dir)->get();
		foreach ($list as $cnt => $cat) {
			$cat->sequence = $cnt + 1;
			$cat->save();
		}
	}
}
