<?php

namespace Modules\Banners\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Modules\Banners\app\Models\Banners;
use Modules\Menu\app\Models\Menu;
use Modules\Mwz\app\Http\Controllers\AdminController;

class BannerAdminController extends AdminController
{
	private $module, $method, $action, $config;

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 15 Jan 2025
	 */
	public function __construct()
	{
		$explode = explode('.', Route::currentRouteName());
		$this->module = $explode[1] ?? 'banners';
		$this->method = $explode[2] ?? 'index';
		$this->action = $explode[3] ?? 'index';

		self::$navbar = [
			['name' => __("banners::banner.{$this->method}.name"), 'url' => route_has("admin.banner.{$this->method}.index")]
		];

		if ($this->method == 'hero') {
			self::$navbar = array_merge([
				['name' => __("banners::banner.{$this->method}.title"), 'url' => null]
			], self::$navbar);
		}

		$this->config = config("banners.{$this->method}");
	}

	/**
	 * Function : branner index
	 * Dev : Petch
	 * Update Date : 15 Jan 2025
	 * @return \Illuminate\Contracts\View\View banners::index
	 */
	public function index()
	{
		$limit = false;
		if (!empty($this->config['setting']['limit']['status']))
			$limit = Banners::where('type', $this->method)->count() >= $this->config['setting']['limit']['max'];
		return view('banners::index', [
			'navbar' => self::$navbar,
			'name' => $this->module,
			'type' => $this->method,
			'config' => $this->config,
			'limit' => $limit
		]);
	}

	/**
	 * Function : datatable ajax
	 * Dev : Petch
	 * Update Date : 15 Jan 2025
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['sequence'];
			foreach ($this->config['table']['header'] as $h) {
				if (!empty($h['status']))
					$dt_column[] = $h['column'];
			}
			if (!empty($this->config['table']['btn']['sort']))
				$dt_column[] = 'sort';
			array_push($dt_column, 'updated_at', 'action');

			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create Branners object
			$req = Banners::where('type', $this->method);

			// filter
			if (!empty($dt_filter['section']))
				$req = $req->where('section', $dt_filter['section']);
			if (!empty($dt_filter['menu_id']))
				$req = $req->where('menu_id', $dt_filter['menu_id']);

			// search 
			if (!empty($dt_search)) {
				$req = $req->whereHas('local', function ($q) use ($dt_search) {
					$q->where('name', 'like', "%{$dt_search}%");
				});
			}

			// count all 
			$dt_total = $req->count();

			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)
				->limit($dt_length)
				->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('menu_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('image', function ($record) {
					switch ($record->style) {
						case 1:
							return self::view_file('image', $record->image);
							break;
						case 2:
							return self::view_file('youtube', $record->youtube);
							break;
						case 3:
							return self::view_file('video', $record->video);
							break;
					}
				})->editColumn('name', function ($record) {
					return str_repeat(' - ', $record->depth) . ($record->local->name ?? '');
				})->editColumn('menu', function ($record) {
					if (!empty($record->menu_id)) {
						return $record->menu->local->name ?? '-';
					} else {
						return __('admin.index');
					}
				})->editColumn('section', function ($record) {
					return   $this->config['table']['filter']['section']['options'][$record->section] ?? '-';
				})->editColumn('view', function ($record) {
					return   $record->view ?? '0';
				})->editColumn('sort', function ($record) use ($dt_order) {
					return self::btn_sort("admin.{$this->module}.{$this->method}.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';

					if (!empty($this->config['table']['btn']['status']))
						$btn .= self::btn_status("admin.{$this->module}.{$this->method}.set_status", $record->id, 'setStatus', $record->status);
					if (!empty($this->config['table']['btn']['edit']))
						$btn .= self::btn_route("admin.{$this->module}.{$this->method}.edit", $record->id);
					if (!empty($this->config['table']['btn']['delete']))
						$btn .= self::btn_delete("admin.{$this->module}.{$this->method}.set_delete", $record->id, 'setDelete');

					$btn .= '</div>';

					return $btn;
				})->escapeColumns([]);
			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : banner form
	 * Dev : Pooh
	 * Update Date : 22 Apr 2024
	 * @param int $id
	 * @return \Illuminate\Contracts\View\View banners::form
	 */
	public function form($id = 0)
	{
		self::$navbar[] =  ['name' => __("action.{$this->action}"), 'url' => null];
		$data = null;
		$limit = false;

		if (!empty($id)) {
			$data = Banners::where([['id', $id], ['type', $this->method]])->first();
			if (empty($data))
				return abort(Response::HTTP_NOT_FOUND);

			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}
		} elseif (!empty($this->config['setting']['limit']['status']))
			$limit = Banners::where('type', $this->method)->count() >= $this->config['setting']['limit']['max'];

		return view('banners::form', [
			'data' => $data,
			'navbar' => self::$navbar,
			'name' => $this->module,
			'type' => $this->method,
			'config' => $this->config,
			'limit' => $limit,
			'action' => $this->action,
		]);
	}

	/**
	 * Function : savae
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function save(Request $request)
	{
		$rule = $msg = [];
		$navbar = $this->config['form']['navbar'];
		$sidebar = $this->config['form']['sidebar'];
		$setting = $this->config['setting'];

		if (!empty($navbar['translate']['status'])) {
			foreach (languages() as $lang => $l) {
				foreach ($navbar['translate']['input'] as $k => $i) {
					if (!empty($i['status']) && !empty($i['validate'])) {
						$rule["{$k}_{$lang}"] = ($k != 'name') ? 'required' : 'required|max:255';
						$msg["{$k}_{$lang}.*"] = __("field.{$k}_placeholder") . ' ' . ($l['name'] ?? '');
					}
				}
			}
		}
		foreach ($sidebar as $s) {
			if (empty($s['status']) || empty($s['input']))
				continue;
			foreach ($s['input'] as $k => $i) {
				if (!empty($i['status']) && !empty($i['validate'])) {
					$rule["{$k}"] = 'required';
					$msg["{$k}.*"] = __("field.{$k}_placeholder");
				}
			}
		}
		// banner style rule
		if ($request->get('style') == 1 && (!empty($navbar['banner']['input']['row_1']['image']['validate']))) {
			if (empty($request->get('id')) || !empty($request->get('image_del')) || (empty($request->get('image_old') && empty($request->get('image')))))
				$rule['image'] = ['required', 'image'];
			else
				$rule['image_old'] = ['required'];

			$msg['image*.required'] = __('field.image_placeholder');
			$msg['image.image'] = __('validation.attributes.error.image.mimetypes');
		}
		if ($request->get('style') == 2 && !empty($navbar['banner']['input']['youtube']['validate'])) {
			$rule['youtube'] = ['required'];
			$msg['youtube.required'] = __('field.youtube_placeholder');
			$msg['youtube.required'] = __('validation.attributes.error.youtube.required');
		}
		if (($request->get('style') == 3 && !empty($navbar['banner']['input']['video']['validate']))) {
			if (empty($request->get('id')) || !empty($request->get('video_del')) || (empty($request->get('video_old') && empty($request->get('video')))))
				$rule['video'] = ['required', 'file', 'mimetypes:video/mp4,video/avi,video/mpeg'];
			else
				$rule['video_old'] = ['required'];

			$msg['video*.required'] = __('field.video_placeholder');
			$msg['video.mimetypes'] = __('validation.attributes.error.video.mimetypes');
			$msg['video.file'] = __('validation.attributes.error.video.file');
		}
		// Check if link starts with http:// or https://
		if ($request->has('link') && !empty($request->get('link'))) {
			if (!preg_match('/^https?:\/\//', $request->get('link'))) {
				return self::response(['msg' => 'ลิงก์ต้องขึ้นต้นด้วย http:// หรือ https://'], Response::HTTP_BAD_REQUEST);
			}
		}

		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_BAD_REQUEST);

		$attr = [];
		foreach ($sidebar as $s) {
			if (empty($s['status']) || empty($s['input']))
				continue;

			foreach ($s['input'] as $k => $i) {
				if (empty($i['status']))
					continue;

				if (!empty($setting["limit_{$k}"]['status'])) {
					$chk = $request->get($k);
					$max = $setting["limit_{$k}"]['limit'][$chk] ?? 0;
					if ($max != 0 && Banners::where([['type', $this->method], ['id', '<>', $request->get('id')], [$k, $chk]])->count() >= $max)
						return self::response(['msg' => __("noti.{$k}_limit", ['max' => $max])], Response::HTTP_BAD_REQUEST);
				}

				$attr[$k] = $k == 'status' ? ($request->get($k) ?? 0) : $request->get($k);
			}
		}

		if (!empty($navbar['banner']['status'])) {
			foreach ($navbar['banner']['input'] as $k_b => $b) {
				if (str_contains($k_b, 'row_') && !empty($b['status'])) {
					foreach ($b as $s_k => $s_b) {
						if ($s_k == 'status')
							continue;

						$chk = $this->save_upload($request, $s_k, str_replace('upload_', '', $s_b['type']));
						if (!$chk['status'])
							return self::response(['msg' => ($s_b['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
						$attr[$s_k] = $chk['result'];
					}
					continue;
				}
				if (empty($b['status']))
					continue;

				if ($k_b == 'youtube' && !empty($request->get('youtube')))
					$attr['youtube'] = self::embed($request->get("youtube"));
				elseif ($k_b == 'video' && !empty($request->get('video'))) {
					$chk = $this->save_upload($request, $k_b, str_replace('upload_', '', $b['type']));
					if (!$chk['status'])
						return self::response(['msg' => ($b['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
					$attr[$k_b] = $chk['result'];
				}
				elseif (str_contains($b['type'], 'upload_')) {
					$chk = $this->save_upload($request, $k_b, str_replace('upload_', '', $b['type']));
					if (!$chk['status'])
						return self::response(['msg' => ($s_b['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
					$attr[$k_b] = $chk['result'];
				} else
					$attr[$k_b] = $request->get($k_b);
			}
		}

		if (empty($request->get('id')))
			$attr["sequence"] = Banners::where([['type', $this->method], ['menu_id', $request->get('menu_id')], ['section', $request->get('section') ?? null]])->max('sequence') + 1;

		if ($this->method == 'video') {
			$attr['link'] = $request->get('link');
		}
		$set = Banners::updateOrCreate(['id' => $request->get('id') ?? 0, 'type' => $this->method], $attr);
		if ($set->save()) {
			if (!empty($navbar['translate']['status']))
				self::setLangs($request, $set, $navbar['translate']['input']);

			return self::response(['msg' => __('noti.success'), 'url' => route("admin.{$this->module}.{$this->method}.index")]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : set translate
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	private function save_upload($request, $name, $type)
	{
		if (check_file_upload($request, $name, $type))
			return ['status' => false, 'msg' => __('noti.mime_image')];

		return ['status' => true, 'result' => self::set_image_upload($request, $name, "public/banner/{$this->method}", "{$type}-")];
	}

	/**
	 * Function : set translate
	 * Dev : Pooh
	 * Update Date : 22 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @param int $id
	 * @return void
	 */
	private function setLangs(Request $request, $row, $config)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($config as $key => $item)
				$attr[$key] = !empty($request->get("{$key}_{$lang}")) ? mwz_setTextString($request->get("{$key}_{$lang}"), $item['type'] == 'texteditor') : null;

			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}

	/**
	 * Function : set status
	 * Dev : Pooh
	 * Update Date : 22 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_status(Request $request)
	{
		$set = Banners::find($request->get('id'));
		$set->status = !$set->status;

		if ($set->save())
			return self::response(['msg' => __('noti.success')]);

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : set delete
	 * Dev : Pooh
	 * Update Date : 22 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_delete(Request $request)
	{
		$set = Banners::find($request->get('id'));
		$path_image = $set->image;
		$path_mobile = $set->mobile;
		$path_video = $set->video;
		$section = $set->section;
		$menu_id = $set->menu_id;

		if ($set->delete()) {
			self::delete_image($path_image);
			self::delete_image($path_mobile);
			self::delete_image($path_video);
			self::re_order('ASC', $section, $menu_id);

			$limit = 0;
			if (!empty($this->config['setting']['limit']['status']))
				$limit = Banners::where('type', $this->method)->count() < $this->config['setting']['limit']['max'];

			return self::response(['msg' => __('noti.delete'), 'limit' => $limit]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : set re order
	 * Dev : Pooh
	 * Update Date : 22 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_re_order(Request $request)
	{
		$sort_json = json_decode($request->get('sort_json'), 1);
		if (!empty($sort_json)) {
			foreach ($sort_json as $id => $sort)
				Banners::find($id)->update(['sequence' => $sort]);

			return self::response(['msg' => __('noti.sort')]);
		}

		return self::response(['msg' => __('noti.no_need_order')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : set sort
	 * Dev : Pooh
	 * Update Date : 22 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_sort(Request $request)
	{
		$node = Banners::find($request->get('id'));
		$move = $request->get('move') ?? 'up';
		$sort = $move == 'up' ? $node->sequence - 1 : $node->sequence + 1;
		$count = Banners::where([['type', $this->method], ['menu_id', $node->menu_id], ['section', $node->section]])->count();

		if ($sort > 0 && $sort <= $count) {
			Banners::where([['type', $this->method], ['menu_id', $node->menu_id], ['section', $node->section], ['id', '<>', $node->id], ['sequence', $sort]])
				->update(['sequence' => $node->sequence]);
			$node->sequence = $sort;
			$node->save();

			return self::response(['msg' =>  __('noti.sort')]);
		}

		return self::response(['msg' => __('noti.no_need_order')], Response::HTTP_BAD_REQUEST);
	}

	/**
	 * Function : re order
	 * Dev : Pooh
	 * Update Date : 22 Apr 2024
	 * @return void
	 */
	public function re_order($dir = 'DESC', $section = null, $menu_id = null)
	{
		$list = Banners::where('type', $this->method);
		if (!empty($section))
			$list = $list->where('section', $section);
		if (!empty($menu_id))
			$list = $list->where('menu_id', $menu_id);

		$list = $list->orderBy('sequence')
			->orderBy('updated_at', $dir)
			->get();
		foreach ($list as $cnt => $cat) {
			$cat->sequence = $cnt + 1;
			$cat->save();
		}
	}

	/**
	 * Function : get menu
	 * Dev : Petch
	 * Update Date : 15 Jan 2025
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function get_menu(Request $request)
	{
		$attr = [];
		if (!empty($this->config['setting']['limit_menu_id'])) {
			foreach (array_keys($this->config['setting']['limit_menu_id']['limit']) as $v)
				$attr[] = $v;
		}

		$lists = Menu::whereIn('id', $attr);
		if (!empty($request->get('search')))
			$lists = $lists->where('name', 'like', `%{$request->get('search')}%`);

		$lists = $lists->withDepth()->defaultOrder()->orderBy('sequence')->get();
		$result = [];
		foreach ($lists as $list)
			$result[] = ['id' => $list->id, 'text' => str_repeat(' - ', $list->depth) . $list->local->name, 'image' =>  ''];

		return self::response(['msg' => __('noti.success'), 'results' => $result]);
	}
}
